package ru.yandex.travel.hotels.extranet.entities

import ru.yandex.travel.hotels.extranet.validation.ValidAnnulationRuleItem
import java.math.BigDecimal
import java.util.UUID
import javax.persistence.AttributeOverride
import javax.persistence.AttributeOverrides
import javax.persistence.Column
import javax.persistence.Embedded
import javax.persistence.Entity
import javax.persistence.EnumType
import javax.persistence.Enumerated
import javax.persistence.Id
import javax.persistence.ManyToOne
import javax.persistence.Table

@Entity
@Table(name = "annulation_rules")
@ValidAnnulationRuleItem
class AnnulationRule(
    @Enumerated(EnumType.STRING) var penaltyType: AnnulationPenaltyType,
    @ManyToOne var policy: AnnulationPolicy? = null,
    @Embedded @AttributeOverrides(
        value = [AttributeOverride(
            name = "amount",
            column = Column(name = "units_amount")
        ), AttributeOverride(name = "unit", column = Column(name = "units_type"))]
    ) var start: AnnulationRuleStart? = null,
    var penaltyNominal: BigDecimal? = null,
) {
    @Id
    val id: UUID = UUID.randomUUID()

    fun comparator(): Int {
        val start = this.start
        return if (start == null) {
            Int.MAX_VALUE
        } else {
            when (start.unit) {
                AnnulationRuleStart.Unit.DAYS -> (start.amount * 24)
                AnnulationRuleStart.Unit.HOURS -> start.amount
            }
        }
    }

    fun describe(): String {
        val start = this.start
        var prefix = if (start != null) {
            if (start.amount == 0) {
                "с момента заезда: "
            } else {
                "начиная с ${start.describe()} до заезда: "
            }
        } else {
            ""
        }
        var descr = when (penaltyType) {
            AnnulationPenaltyType.NONE -> "без штрафа"
            AnnulationPenaltyType.NIGHTS -> if (penaltyNominal == BigDecimal.valueOf(1)) {
                "штраф — первая ночь"
            } else {
                "штраф — ${penaltyNominal?.toInt()} ночей"
            }
            AnnulationPenaltyType.PERCENTAGE -> "штраф — ${penaltyNominal?.toLong()}%"
            AnnulationPenaltyType.FIXED -> "штраф — ${penaltyNominal?.setScale(2)?.toFloat()}"
            AnnulationPenaltyType.FULL -> "штраф — вся стоимость бронирования"
        }
        return "$prefix$descr"
    }
}

