package ru.yandex.travel.hotels.extranet.extract

import org.springframework.batch.core.Job
import org.springframework.batch.core.Step
import org.springframework.batch.core.configuration.annotation.DefaultBatchConfigurer
import org.springframework.batch.core.configuration.annotation.EnableBatchProcessing
import org.springframework.batch.core.configuration.annotation.JobBuilderFactory
import org.springframework.batch.core.configuration.annotation.StepBuilderFactory
import org.springframework.batch.core.configuration.annotation.StepScope
import org.springframework.batch.item.database.JpaItemWriter
import org.springframework.batch.item.database.builder.JpaItemWriterBuilder
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.beans.factory.annotation.Value
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.context.annotation.Import
import org.springframework.context.annotation.Primary
import org.springframework.orm.jpa.JpaTransactionManager
import ru.yandex.travel.hotels.extranet.TBankOrderInfo
import ru.yandex.travel.hotels.extranet.TDBoyOrder
import ru.yandex.travel.hotels.extranet.entities.orders.BankOrder
import ru.yandex.travel.hotels.extranet.entities.orders.BoYOrder
import ru.yandex.travel.hotels.extranet.extract.grpc.BankOrderIntermediateProcessor
import ru.yandex.travel.hotels.extranet.extract.grpc.BankOrdersGrpcReader
import ru.yandex.travel.hotels.extranet.extract.grpc.FROM_PROP
import ru.yandex.travel.hotels.extranet.extract.grpc.OrderIntermediateProcessor
import ru.yandex.travel.hotels.extranet.extract.grpc.OrdersGrpcClientFactory
import ru.yandex.travel.hotels.extranet.extract.grpc.OrdersGrpcProperties
import ru.yandex.travel.hotels.extranet.extract.grpc.OrdersGrpcReader
import ru.yandex.travel.hotels.extranet.extract.grpc.TILL_PROP
import ru.yandex.travel.hotels.extranet.extract.grpc.UpdateDateRunIncrementer
import ru.yandex.travel.hotels.extranet.extract.support.ErrorReportingListener
import ru.yandex.travel.hotels.extranet.extract.support.MasterLockConfiguration
import ru.yandex.travel.hotels.extranet.extract.support.ParametersValidator
import ru.yandex.travel.hotels.extranet.extract.support.bankorders.BankOrderDetailsWebViewUpdater
import ru.yandex.travel.hotels.extranet.service.IManualJobOverrideService
import java.util.Date
import javax.persistence.EntityManagerFactory
import javax.sql.DataSource

@Configuration
@Import(MasterLockConfiguration::class)
@EnableBatchProcessing
@EnableConfigurationProperties(BatchProperties::class)
@ConditionalOnProperty(value = ["spring.batch.job.enabled"], havingValue = "true", matchIfMissing = true)
open class BatchConfiguration(
    private val jobFactory: JobBuilderFactory,
    private val stepFactory: StepBuilderFactory,
    private val entityFactory: EntityManagerFactory,
    private val apiProperties: OrdersGrpcProperties,
    private val grpcClientFactory: OrdersGrpcClientFactory,
    private val batchProperties: BatchProperties,
    private val overrideService: IManualJobOverrideService,
    dataSource: DataSource,
) : DefaultBatchConfigurer(dataSource) {

    @Bean
    @StepScope
    open fun ordersReader(@Value("#{jobParameters}") params: Map<String, Date>): OrdersGrpcReader {
        return OrdersGrpcReader(
            grpcClientFactory, apiProperties,
            params[FROM_PROP]!!.toInstant(),
            params.getOrDefault(TILL_PROP, Date()).toInstant()
        )
    }

    @Bean
    open fun ordersProcessor(): OrderIntermediateProcessor {
        return OrderIntermediateProcessor()
    }

    @Bean
    open fun ordersWriter(): JpaItemWriter<BoYOrder> {
        return JpaItemWriterBuilder<BoYOrder>()
            .entityManagerFactory(entityFactory)
            .build()
    }

    @Bean
    open fun orderJob(
        @Qualifier("downloadAndStoreOrders") step: Step,
        bankOrdersItemWriteListener: BankOrderDetailsWebViewUpdater
    ): Job {
        val jobName = "Sync Orders Job"
        return jobFactory.get(jobName)
            .incrementer(UpdateDateRunIncrementer(apiProperties, jobRepository, jobName, overrideService))
            .listener(ErrorReportingListener())
            .validator(ParametersValidator())
            .flow(step)
            // when something odd happens
            .on("UNKNOWN").end("STOPPED")
            .on("FAILED").fail()
            .on("*").end()
            .end()
            .build()
    }

    @Bean
    open fun downloadAndStoreOrders(reader: OrdersGrpcReader): Step {
        return stepFactory.get("Read and save DBoY orders")
            // not obvious, but a new transaction manager is needed here.
            // See https://stackoverflow.com/questions/22509529/jpaitemwriter-no-transaction-is-in-progress
            .transactionManager(JpaTransactionManager(entityFactory))
            .chunk<TDBoyOrder, BoYOrder>(batchProperties.chunkSize)
            .reader(reader)
            .processor(ordersProcessor())
            .writer(ordersWriter())
            .build()
    }

    // ================== Job 2 ============================

    @Bean
    @StepScope
    open fun bankOrderReader(@Value("#{jobParameters}") params: Map<String, Date>): BankOrdersGrpcReader {
        return BankOrdersGrpcReader(
            grpcClientFactory, apiProperties,
            params[FROM_PROP]!!.toInstant(),
            params.getOrDefault(TILL_PROP, Date()).toInstant()
        )
    }

    @Bean
    open fun bankOrdersProcessor(): BankOrderIntermediateProcessor {
        return BankOrderIntermediateProcessor()
    }

    @Bean
    open fun bankOrdersWriter(): JpaItemWriter<BankOrder> {
        return JpaItemWriterBuilder<BankOrder>()
            .entityManagerFactory(entityFactory)
            .build()
    }

    @Bean
    open fun bankOrderJob(
        @Qualifier("downloadAndStoreBankOrders") step: Step,
        bankOrdersItemWriteListener: BankOrderDetailsWebViewUpdater
    ): Job {
        val jobName = "Sync Bank Info"
        return jobFactory.get(jobName)
            .incrementer(UpdateDateRunIncrementer(apiProperties, jobRepository, jobName, overrideService))
            .listener(ErrorReportingListener())
            .listener(bankOrdersItemWriteListener)
            .validator(ParametersValidator())
            .flow(step)
            .end()
            .build()
    }

    @Bean
    open fun downloadAndStoreBankOrders(
        reader: BankOrdersGrpcReader,
        bankOrdersItemWriteListener: BankOrderDetailsWebViewUpdater
    ): Step {
        return stepFactory.get("Read and save bank info")
            // not obvious, but a new transaction manager is needed here.
            // See https://stackoverflow.com/questions/22509529/jpaitemwriter-no-transaction-is-in-progress
            .transactionManager(JpaTransactionManager(entityFactory))
            .chunk<TBankOrderInfo, BankOrder>(batchProperties.chunkSize)
            .reader(reader)
            .processor(bankOrdersProcessor())
            .writer(bankOrdersWriter())
            .listener(bankOrdersItemWriteListener)
            .build()
    }

    @Bean
    @Primary
    open fun appTransactionManager(): JpaTransactionManager {
        return JpaTransactionManager(entityFactory)
    }
}
