package ru.yandex.travel.hotels.extranet.extract.grpc

import com.google.protobuf.Timestamp
import org.springframework.batch.item.ExecutionContext
import org.springframework.batch.item.support.AbstractItemCountingItemStreamItemReader
import ru.yandex.travel.commons.proto.ProtoUtils
import ru.yandex.travel.hotels.extranet.OrdersHotelsExtranetDataServiceGrpc
import ru.yandex.travel.hotels.extranet.TGetDBoyReq
import ru.yandex.travel.hotels.extranet.TGetDBoyRsp
import java.time.Instant
import java.util.concurrent.TimeUnit

abstract class AbstractGrpcReader<Item>(
    private val clientFactory: OrdersGrpcClientFactory,
    private var apiProperties: OrdersGrpcProperties,
    private var from: Instant = apiProperties.startAt,
    private var to: Instant = Instant.now(),
) : AbstractItemCountingItemStreamItemReader<Item>() {
    private var channel: OrdersHotelsExtranetDataServiceGrpc.OrdersHotelsExtranetDataServiceBlockingStub? = null

    private var response: TGetDBoyRsp? = null
    private var iterator: Iterator<Item>? = null

    init {
        this.setName("OrdersGrpcReader")
    }

    private fun query(from: Instant, to: Instant): TGetDBoyRsp {
        val protoReq = TGetDBoyReq.newBuilder()
        protoReq.limit = apiProperties.limit
        protoReq.updatedAtFrom = ProtoUtils.fromInstant(from)
        protoReq.updatedAtTo = ProtoUtils.fromInstant(to)
        return grpcFunction(protoReq, channel!!.withDeadlineAfter(1, TimeUnit.MINUTES))
    }

    @Synchronized
    override fun doRead(): Item? {
        queryNextPageIfNeeded()
        return if (iterator?.hasNext() == true) iterator?.next() else null
    }

    private fun queryNextPageIfNeeded() {
        if (response == null || (!iterator!!.hasNext() && response!!.hasMore)) {
            response = query(from, to)
            iterator = getList(response!!).iterator()
            val last = getList(response!!).lastOrNull()
            if (last != null) from = getUpdatedAt(last).instant()
        }
    }

    protected abstract fun grpcFunction(
        protoReq: TGetDBoyReq.Builder,
        service: OrdersHotelsExtranetDataServiceGrpc.OrdersHotelsExtranetDataServiceBlockingStub
    ): TGetDBoyRsp

    protected abstract fun getUpdatedAt(last: Item): Timestamp

    protected abstract fun getList(response: TGetDBoyRsp): List<Item>

    override fun doOpen() {
        channel = clientFactory.createExtranetService()
    }

    override fun open(executionContext: ExecutionContext) {
        super.open(executionContext)
        executionContext.get(FROM_PROP)?.let { from = it as Instant }
        executionContext.get(TILL_PROP)?.let { to = it as Instant }
    }

    @Synchronized
    override fun doClose() {
        channel = null
        response = null
        iterator = null
    }
}
