package ru.yandex.travel.hotels.extranet.extract.grpc

import io.grpc.ClientInterceptor
import io.grpc.ManagedChannelBuilder
import io.opentracing.Tracer
import lombok.SneakyThrows
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper
import ru.yandex.travel.hotels.extranet.OrdersHotelsExtranetDataServiceGrpc
import ru.yandex.travel.hotels.extranet.OrdersHotelsExtranetDataServiceGrpc.OrdersHotelsExtranetDataServiceBlockingStub
import ru.yandex.travel.hotels.proto.PromoServiceV1Grpc
import ru.yandex.travel.orders.client.ChannelState
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory
import ru.yandex.travel.orders.client.HAGrpcChannelFactory
import ru.yandex.travel.orders.client.LabeledChannel
import ru.yandex.travel.orders.proto.EServerState
import ru.yandex.travel.orders.proto.HADiscoveryInterfaceV1Grpc
import ru.yandex.travel.orders.proto.TPingRpcReq
import ru.yandex.travel.tvm.TvmWrapper
import java.net.InetAddress
import java.util.Objects

@Configuration
@EnableConfigurationProperties(
    OrdersGrpcProperties::class
)
@ConditionalOnProperty(prefix = "job.scheduling", name = ["enabled"], havingValue = "true", matchIfMissing = true)
open class GrpcConfiguration(private val properties: OrdersGrpcProperties) {

    @Bean
    open fun ordersGrpcChannelFactory(
        @Autowired(required = false) tvm: TvmWrapper?,
        @Autowired(required = false) tracer: Tracer?,
    ): HAGrpcChannelFactory {
        return HAGrpcChannelFactory.Builder.newBuilder()
            .withPingProducer {
                FutureUtils.buildCompletableFuture(
                    HADiscoveryInterfaceV1Grpc
                        .newFutureStub(it)
                        .ping(TPingRpcReq.newBuilder().build())
                ).thenApply { rsp ->
                    when (rsp.state) {
                        EServerState.SS_MASTER -> ChannelState.READY_MASTER
                        else -> ChannelState.READY
                    }
                }
            }
            .withFailureDetectorProperties(properties.failureDetection)
            .withChannelSupplier(GrpcChannelSupplierFactory(properties).channelSupplier)
            .withChannelBuilder { label ->
                this.createChannel(
                    label,
                    getTvmHelper(tvm),
                    tracer
                )
            }
            .build()
    }

    @SneakyThrows
    private fun createChannel(
        target: String,
        tvmHeaderClientHelper: TvmHeaderClientHelper?,
        tracer: Tracer?,
    ): LabeledChannel {
        val clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().canonicalHostName)
        val loggingClientInterceptor = LoggingClientInterceptor(
            clientFqdn, target, setOf(PromoServiceV1Grpc.getPingMethod().fullMethodName), tracer
        )
        val defaultTimeoutClientInterceptor = DefaultTimeoutClientInterceptor(
            properties.timeout
        )
        val interceptors: MutableList<ClientInterceptor> = arrayListOf(
            loggingClientInterceptor, defaultTimeoutClientInterceptor,
        )
        if (tracer != null) {
            interceptors.add(TracerClientInterceptor(tracer))
        }
        if (tvmHeaderClientHelper != null) {
            interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.tvm?.destinationAlias))
        }
        return LabeledChannel(
            target,
            ManagedChannelBuilder
                .forTarget(target)
                .intercept(interceptors)
                .usePlaintext()
                .maxInboundMessageSize(properties.maxMessageSize)
                .build()
        )
    }

    @Bean
    open fun ordersClientFactory(
        channelFactory: HAGrpcChannelFactory
    ): OrdersGrpcClientFactory {
        return object : OrdersGrpcClientFactory {
            override fun createExtranetService(): OrdersHotelsExtranetDataServiceBlockingStub {
                return OrdersHotelsExtranetDataServiceGrpc.newBlockingStub(channelFactory.roundRobinChannel)
            }
        }
    }

    private fun getTvmHelper(tvm: TvmWrapper?): TvmHeaderClientHelper? {
        return if (properties.tvm?.enabled == true && tvm != null) {
            TvmHeaderClientHelper(tvm)
        } else {
            null
        }
    }
}
