package ru.yandex.travel.hotels.extranet.extract.grpc

import com.google.protobuf.Timestamp
import org.springframework.batch.item.ItemProcessor
import ru.yandex.travel.commons.proto.ProtoUtils
import ru.yandex.travel.commons.proto.ProtoUtils.toInstant
import ru.yandex.travel.commons.proto.TPrice
import ru.yandex.travel.hotels.extranet.TBankOrderTransactionInfo
import ru.yandex.travel.hotels.extranet.TDBoyOrder
import ru.yandex.travel.hotels.extranet.TFinancialEventInfo
import ru.yandex.travel.hotels.extranet.entities.HotelAgreement
import ru.yandex.travel.hotels.extranet.entities.HotelIdentifier
import ru.yandex.travel.hotels.extranet.entities.orders.BillingTransaction
import ru.yandex.travel.hotels.extranet.entities.orders.BoYOrder
import ru.yandex.travel.hotels.extranet.entities.orders.FinancialEvent
import ru.yandex.travel.hotels.extranet.entities.orders.Guest
import ru.yandex.travel.hotels.extranet.entities.orders.GuestOrderIdx
import java.math.BigDecimal
import java.time.Instant

class OrderIntermediateProcessor : ItemProcessor<TDBoyOrder, BoYOrder> {
    override fun process(proto: TDBoyOrder): BoYOrder {
        return protoToEntityOrder(proto)
    }

    private fun protoToEntityFinancialEvent(proto: TFinancialEventInfo, order: BoYOrder): FinancialEvent {
        val financialEvent = FinancialEvent(
            id = proto.financialEventId,
            order = order,
            billingClientId = proto.billingClientId,
            billingContractId = proto.billingContractId,
            partnerAmount = proto.partnerAmount.toBigDecimal(),
            feeAmount = proto.feeAmount.toBigDecimal(),
            promoCodePartnerAmount = proto.promoCodePartnerAmount.toBigDecimalNullable(),
            promoCodeFeeAmount = proto.promoCodeFeeAmount.toBigDecimalNullable(),
            plusPartnerAmount = proto.plusPartnerAmount.toBigDecimalNullable(),
            plusFeeAmount = proto.plusFeeAmount.toBigDecimalNullable(),
            payoutAt = proto.payoutAt.instantNullable()
        )
        financialEvent.billingTransactions = proto.bankOrderTransactionsList.map {
            protoToEntityBillingTransaction(it, financialEvent)
        }.toMutableList()
        return financialEvent
    }

    private fun protoToEntityBillingTransaction(
        proto: TBankOrderTransactionInfo,
        financialEvent: FinancialEvent
    ): BillingTransaction {
        return BillingTransaction(
            id = proto.id,
            transactionType = proto.transactionType,
            paymentType = proto.paymentType,
            value = proto.valueAmount.toBigDecimal(),
            createdAtBillingTransaction = proto.createdAt.instant(),
            payoutAt = proto.payoutAt.instantNullable(),
            accountingActAt = proto.accountingActAt.instantNullable(),
            exportedToYt = proto.exportedToYt,
            actCommitted = proto.actCommitted,
            ytId = proto.ytId,
            financialEvent = financialEvent
        )
    }

    private fun protoToEntityOrder(proto: TDBoyOrder): BoYOrder {
        val order = BoYOrder(
            id = proto.travelOrderGuid,
            prettyId = proto.travelPrettyOrderId,
            partnerOrderId = proto.partnerOrderId,
            hotelOrderState = proto.state,
            orderCreatedAt = proto.createdAt.instant(),
            orderUpdatedAt = proto.updatedAt.instant(),
            orderCancelledAt = proto.cancellationDateTime.instantNullable(),
            checkInDate = ProtoUtils.toLocalDate(proto.checkInDate),
            checkOutDate = ProtoUtils.toLocalDate(proto.checkOutDate),
            hotelPrice = proto.hotelPrice.toBigDecimal(),
            fiscalPrice = proto.fiscalPrice.toBigDecimal(),
            discount = proto.discount.toBigDecimalNullable(),
            paymentInfo = PaymentInfo(
                useDeferred = proto.paymentInfo.useDeferred,
                paymentState = proto.paymentInfo.paymentState
            ),
            costAfterReservation = proto.costAfterReservation.toBigDecimalNullable()
        )

        order.financialEvents = proto.financialEventInfoList.map {
            protoToEntityFinancialEvent(it, order)
        }.toMutableList()

        order.guests = proto.guestsList.mapIndexed { index, it ->
            Guest(
                id = GuestOrderIdx(order, index + 1),
                firstName = it.firstName,
                lastName = it.lastName,
                isChild = it.isChild,
                age = if (it.age == 0) null else it.age,
            )
        }.toMutableList()

        proto.hotelAgreement.let {
            order.hotelAgreement = HotelAgreement(
                id = it.id,
                hotelIdentifier = HotelIdentifier(it.partnerId, it.hotelId),
                financialClientId = it.financialClientId,
                financialContractId = it.financialContractId,
                orderConfirmedRate = it.orderConfirmedRate,
                orderRefundedRate = it.orderRefundedRate,
                agreementEndDate = Instant.ofEpochMilli(it.agreementEndDate),
                agreementStartDate = Instant.ofEpochMilli(it.agreementStartDate),
                fullLegalName = it.fullLegalName,
                externalHotelId = it.externalHotelId.ifEmpty { null },
            )

        }
        return order
    }
}

fun TPrice.toBigDecimal(): BigDecimal {
    return BigDecimal.valueOf(this.amount, this.precision)
}

fun TPrice.toBigDecimalNullable(): BigDecimal? {
    return if (this == TPrice.getDefaultInstance()) return null
    else BigDecimal.valueOf(this.amount, this.precision)
}

fun Timestamp.instant(): Instant {
    return toInstant(this)
}

fun Timestamp.instantNullable(): Instant? {
    return if (this == Timestamp.getDefaultInstance()) null
    else toInstant(this)
}
