package ru.yandex.travel.hotels.extranet.extract.support

import org.slf4j.LoggerFactory
import org.springframework.batch.core.BatchStatus
import org.springframework.batch.core.ExitStatus
import org.springframework.batch.core.explore.JobExplorer
import org.springframework.batch.core.launch.JobExecutionNotRunningException
import org.springframework.batch.core.launch.JobLauncher
import org.springframework.batch.core.launch.JobOperator
import org.springframework.batch.core.launch.NoSuchJobException
import org.springframework.batch.core.repository.JobRepository
import org.springframework.beans.factory.annotation.Value
import org.springframework.boot.autoconfigure.batch.JobLauncherApplicationRunner
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.scheduling.annotation.EnableScheduling
import org.springframework.scheduling.annotation.Scheduled
import org.springframework.stereotype.Component
import ru.yandex.travel.workflow.ha.MasterStatusAwareResource
import java.util.Date
import java.util.concurrent.atomic.AtomicBoolean
import javax.annotation.PreDestroy
import javax.transaction.Transactional

/**
 * The bean that starts the job.
 *
 * @implNote: it extends JobLauncherApplicationRunner not because it's necessary to implement the interfaces,
 * but because there's a lot of functionality we'd like to borrow.
 */
@Component
@EnableScheduling
@ConditionalOnProperty(prefix = "job.scheduling", name = ["enabled"], havingValue = "true", matchIfMissing = true)
open class JobScheduleRunner(
    jobLauncher: JobLauncher,
    private val jobExplorer: JobExplorer,
    private val jobRepository: JobRepository,
    @Value("\${high-availability.enabled:false}")
    private val masterLockEnabled: Boolean,
    private val jobOperator: JobOperator,
) : MasterStatusAwareResource, JobLauncherApplicationRunner(jobLauncher, jobExplorer, jobRepository) {

    private val log = LoggerFactory.getLogger(javaClass)

    private val master = AtomicBoolean(false)

    @Scheduled(fixedDelayString = "\${job.scheduling.fixedDelay:60000}", initialDelay = 1000)
    fun launchJob() {
        if (master.get() ||
            // we don't have opportunity to run locks on h2
            !masterLockEnabled
        ) {
            launchJobFromProperties(null)
        }
    }

    override fun run(vararg args: String?) {
        // do nothing as we don't want the bean to be launched as ApplicationRunner
    }

    override fun promotedToMaster() {
        stopAll()
        master.set(true)
    }

    override fun prepareToStandby() {
        master.set(false)
        stopAll()
    }

    override fun forceStandby() {
        prepareToStandby()
        stopAll()
    }

    @PreDestroy
    @Transactional(Transactional.TxType.REQUIRES_NEW)
    override fun stopAll() {
        log.warn("Stopping all jobs")
        jobExplorer.jobNames.forEach { name ->
            try {
                log.debug("Looking for a running job {}", name)
                val executions = jobOperator.getRunningExecutions(name)
                executions.forEach { execution ->
                    log.warn("Found existing job: {}, killing it.", execution)
                    try {
                        jobOperator.stop(execution)
                    } catch (e: JobExecutionNotRunningException) {
                        // it's fine
                        // it can happen when an app was terminated unexpectedly during the process of stopping
                        log.warn("Can't stop the job: {}, force stopping it.", execution)
                        forceStop(execution)
                    }
                }
            } catch (ignored: NoSuchJobException) {
                log.info("No running jobs. {}", ignored.message)
            } catch (e: Exception) {
                log.error("Error stopping the job", e)
                // TODO report to solomon. Error here is critical
            }
        }
    }

    private fun forceStop(execution: Long) {
        jobExplorer.getJobExecution(execution)!!.let {
            it.stepExecutions.forEach { step ->
                step.status = BatchStatus.STOPPED
                step.exitStatus = ExitStatus.STOPPED
                step.endTime = Date()
            }
            it.status = BatchStatus.STOPPED
            it.endTime = Date()
            it.exitStatus = ExitStatus.STOPPED
            jobRepository.update(it)
        }
    }
}
