package ru.yandex.travel.hotels.extranet.extract.support.bankorders

import org.slf4j.LoggerFactory
import org.springframework.batch.core.ItemWriteListener
import org.springframework.batch.core.JobExecution
import org.springframework.batch.core.JobExecutionListener
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional
import ru.yandex.travel.hotels.extranet.entities.orders.BankOrder
import ru.yandex.travel.hotels.extranet.entities.orders.BankOrderDetailWebView
import ru.yandex.travel.hotels.extranet.entities.orders.BankOrderDetailsWebViewIdx
import ru.yandex.travel.hotels.extranet.entities.orders.BoYOrderWebView
import ru.yandex.travel.hotels.extranet.repository.BankOrderDetailsWebViewRepository
import ru.yandex.travel.hotels.extranet.repository.BankOrdersRepository
import ru.yandex.travel.hotels.extranet.repository.OrdersWebViewRepository
import java.math.BigDecimal

/**
 * Responsible for updating the table `bank_order_details_web_view`.
 */
@Component
open class BankOrderDetailsWebViewUpdater(
    private val orderRepo: OrdersWebViewRepository,
    private val viewsRepo: BankOrderDetailsWebViewRepository,
    private val bankOrderRepo: BankOrdersRepository,
) : JobExecutionListener, ItemWriteListener<BankOrder> {

    private val log = LoggerFactory.getLogger(javaClass)

    @Throws(Exception::class)
    @Transactional
    override fun afterWrite(items: List<BankOrder?>?) {
        log.info("Running after update for ${items?.size ?: 0} items")
        log.debug(" updating ${items?.map { it?.bankOrderId }}")
        items?.filter { it != null }
            ?.map {
                it as BankOrder
            }?.forEach { calculateForBankOrder(it) }
        viewsRepo.flush()
        log.debug(" done updating")
    }

    private fun calculateForBankOrder(bankOrder: BankOrder) {
        val calculated = bankOrder.bankOrderDetails.map {
            val order: BoYOrderWebView = orderRepo.findByYtId(it.ytId) ?: return@map null
            BankOrderDetailWebView(
                id = BankOrderDetailsWebViewIdx(
                    paymentBatchId = it.bankOrder.paymentBatchId,
                    bankOrderId = it.bankOrder.bankOrderId,
                    orderId = order.id,
                    transactionType = it.transactionType,
                ),
                hotelId = order.hotelId,
                prettyId = order.prettyId,
                paidAmount = it.sum,
                agencyCommission = it.agencyCommission,
                guestFirstName = order.guestFirstName,
                guestLastName = order.guestLastName,
                fiscalPrice = order.fiscalPrice,
                hotelPrice = order.hotelPrice,
                orderCreatedAt = order.createdAt,
                checkInDate = order.checkInDate,
                checkOutDate = order.checkOutDate,
            )
        }.filterNotNull()

        if (calculated.isEmpty()) {
            // might be a dolphin order. ATM we don't have a reliable way to filter it on orc side.
            log.info("Didn't find an order for bankOrder: " + bankOrder.paymentBatchId)

            bankOrder.calculateAttempts++
            bankOrderRepo.save(bankOrder)
            return
        }

        val overallSum = calculated
            .groupBy { it.id }
            .values
            .map {
                it.reduce { acc, el ->
                    acc.paidAmount = acc.paidAmount.plus(el.paidAmount)
                    acc.agencyCommission = (acc.agencyCommission ?: BigDecimal.ZERO).plus(el.agencyCommission ?: BigDecimal.ZERO)
                    return@reduce acc
                }
            }.map {
                viewsRepo.save(it)
                it.signedPaidAmount
            }.sumOf { it!! }

        if (overallSum.compareTo(bankOrder.sum) != 0 && bankOrder.isFetched) {
            // this means that some assumptions on how the app works are wrong, it needs close attention of developers
            log.error(
                "The sum for bank order details didn't match the full sum in bank order ${bankOrder.bankOrderId}, " +
                    "${bankOrder.paymentBatchId}: $overallSum <> ${bankOrder.sum}"
            )
        }
    }

    @Transactional
    override fun afterJob(jobExecution: JobExecution) {
        log.debug("Starting the before job hook.")
        val forUpdate = viewsRepo.findBankOrderForUpdate()
        log.debug("For update bank orders: updating ${forUpdate.size}")
        forUpdate.forEach { calculateForBankOrder(it) }
        log.debug("Done updating planned for updating, starting working on missing ones")

        val notUpdated = viewsRepo.findMissing()
        log.debug("Missing bank orders: updating ${notUpdated.size}")
        notUpdated.forEach { calculateForBankOrder(it) }

        log.debug("Finished after job hook.")
    }

    override fun beforeJob(jobExecution: JobExecution) {
    }

    override fun beforeWrite(items: MutableList<out BankOrder>) {
    }

    override fun onWriteError(exception: java.lang.Exception, items: MutableList<out BankOrder>) {
        // probably useless statement
        log.error("Error on Bank order saving", exception)
    }
}
