package ru.yandex.travel.hotels.extranet.grpc

import io.grpc.stub.StreamObserver
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.travel.grpc.GrpcService
import ru.yandex.travel.hotels.extranet.AddRatePlanRequest
import ru.yandex.travel.hotels.extranet.AddRatePlanResponse
import ru.yandex.travel.hotels.extranet.DeleteRatePlanRequest
import ru.yandex.travel.hotels.extranet.DeleteRatePlanResponse
import ru.yandex.travel.hotels.extranet.DisableRatePlanRequest
import ru.yandex.travel.hotels.extranet.DisableRatePlanResponse
import ru.yandex.travel.hotels.extranet.EnableRatePlanRequest
import ru.yandex.travel.hotels.extranet.EnableRatePlanResponse
import ru.yandex.travel.hotels.extranet.GetRatePlanRequest
import ru.yandex.travel.hotels.extranet.GetRatePlanResponse
import ru.yandex.travel.hotels.extranet.HideRatePlanRequest
import ru.yandex.travel.hotels.extranet.HideRatePlanResponse
import ru.yandex.travel.hotels.extranet.ListRatePlansRequest
import ru.yandex.travel.hotels.extranet.ListRatePlansResponse
import ru.yandex.travel.hotels.extranet.RatePlanServiceGrpc
import ru.yandex.travel.hotels.extranet.ShowRatePlanRequest
import ru.yandex.travel.hotels.extranet.ShowRatePlanResponse
import ru.yandex.travel.hotels.extranet.UpdateRatePlanRequest
import ru.yandex.travel.hotels.extranet.UpdateRatePlanResponse
import ru.yandex.travel.hotels.extranet.service.content.rateplans.RatePlanService

@GrpcService(authenticateService = true, authenticateUser = true)
class RatePlanGrpcServiceImpl @Autowired constructor(private val ratePlanService: RatePlanService) :
    RatePlanServiceGrpc.RatePlanServiceImplBase() {

    override fun listRatePlans(
        request: ListRatePlansRequest, responseObserver: StreamObserver<ListRatePlansResponse>
    ) {
        grpcWrapper(request, responseObserver) {
            ListRatePlansResponse.newBuilder().addAllRatePlans(
                ratePlanService.listRatePlans(
                    request.hotelId,
                    includeDisabled = request.includeDisabled,
                    includeHidden = request.includeHidden
                )
            ).build()
        }
    }

    override fun getRatePlan(request: GetRatePlanRequest, responseObserver: StreamObserver<GetRatePlanResponse>) {
        grpcWrapper(request, responseObserver) {
            GetRatePlanResponse.newBuilder()
                .setRatePlan(ratePlanService.getRatePlan(request.hotelId, request.ratePlanId)).build()
        }
    }

    override fun addRatePlan(request: AddRatePlanRequest, responseObserver: StreamObserver<AddRatePlanResponse>) {
        grpcWrapper(request, responseObserver) {
            AddRatePlanResponse.newBuilder().setRatePlan(
                ratePlanService.addRatePlan(
                    request.hotelId,
                    request.name,
                    request.annulationPolicyId,
                    request.description,
                    request.roomCategoryIdsList.toSet()
                )
            ).build()
        }
    }

    override fun updateRatePlan(
        request: UpdateRatePlanRequest, responseObserver: StreamObserver<UpdateRatePlanResponse>
    ) {
        grpcWrapper(request, responseObserver) {
            UpdateRatePlanResponse.newBuilder().setRatePlan(
                ratePlanService.updateRatePlan(
                    request.hotelId,
                    request.ratePlanId,
                    request.name.ifEmpty { null },
                    annulationPolicyId = if (request.annulationPolicyId != 0L) {
                        request.annulationPolicyId
                    } else null,
                    description = if (request.clearDescription) {
                        ""
                    } else {
                        request.description.ifEmpty { null }
                    },
                    roomCategories = if (request.clearCategories) {
                        emptySet()
                    } else {
                        request.roomCategoryIdsList.toSet().ifEmpty { null }
                    }
                )
            ).build()
        }
    }

    override fun deleteRatePlan(
        request: DeleteRatePlanRequest, responseObserver: StreamObserver<DeleteRatePlanResponse>
    ) {
        grpcWrapper(request, responseObserver) {
            ratePlanService.deleteRatePlan(request.hotelId, request.ratePlanId)
            DeleteRatePlanResponse.getDefaultInstance()
        }
    }

    override fun hideRatePlan(request: HideRatePlanRequest, responseObserver: StreamObserver<HideRatePlanResponse>) {
        grpcWrapper(request, responseObserver) {
            ratePlanService.hideRatePlan(request.hotelId, request.ratePlanId)
            HideRatePlanResponse.getDefaultInstance()
        }
    }

    override fun showRatePlan(request: ShowRatePlanRequest, responseObserver: StreamObserver<ShowRatePlanResponse>) {
        grpcWrapper(request, responseObserver) {
            ratePlanService.showRatePlan(request.hotelId, request.ratePlanId)
            ShowRatePlanResponse.getDefaultInstance()
        }
    }

    override fun disableRatePlan(
        request: DisableRatePlanRequest, responseObserver: StreamObserver<DisableRatePlanResponse>
    ) {
        grpcWrapper(request, responseObserver) {
            ratePlanService.disableRatePlan(request.hotelId, request.ratePlanId)
            DisableRatePlanResponse.getDefaultInstance()
        }
    }

    override fun enableRatePlan(
        request: EnableRatePlanRequest, responseObserver: StreamObserver<EnableRatePlanResponse>
    ) {
        grpcWrapper(request, responseObserver) {
            ratePlanService.enableRatePlan(request.hotelId, request.ratePlanId)
            EnableRatePlanResponse.getDefaultInstance()
        }
    }
}
