package ru.yandex.travel.hotels.extranet.grpc

import com.google.common.base.Preconditions
import io.grpc.stub.StreamObserver
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.travel.grpc.GrpcService
import ru.yandex.travel.hotels.extranet.AddRoomCategoryRequest
import ru.yandex.travel.hotels.extranet.AddRoomCategoryResponse
import ru.yandex.travel.hotels.extranet.DeleteRoomCategoryRequest
import ru.yandex.travel.hotels.extranet.DeleteRoomCategoryResponse
import ru.yandex.travel.hotels.extranet.DisableRoomCategoryRequest
import ru.yandex.travel.hotels.extranet.DisableRoomCategoryResponse
import ru.yandex.travel.hotels.extranet.EnableRoomCategoryRequest
import ru.yandex.travel.hotels.extranet.EnableRoomCategoryResponse
import ru.yandex.travel.hotels.extranet.GetRoomCategoryRequest
import ru.yandex.travel.hotels.extranet.GetRoomCategoryResponse
import ru.yandex.travel.hotels.extranet.HideRoomCategoryRequest
import ru.yandex.travel.hotels.extranet.HideRoomCategoryResponse
import ru.yandex.travel.hotels.extranet.ListRoomCategoriesRequest
import ru.yandex.travel.hotels.extranet.ListRoomCategoriesResponse
import ru.yandex.travel.hotels.extranet.RoomCategoriesServiceGrpc
import ru.yandex.travel.hotels.extranet.ShowRoomCategoryRequest
import ru.yandex.travel.hotels.extranet.ShowRoomCategoryResponse
import ru.yandex.travel.hotels.extranet.service.content.roomcat.RoomCategoryService
import java.io.Closeable
import javax.persistence.EntityManagerFactory

@GrpcService(authenticateService = true, authenticateUser = true)
class RoomCategoryGrpcServiceImpl @Autowired constructor(
    private val service: RoomCategoryService,
    private val entityFactory: EntityManagerFactory,
) : RoomCategoriesServiceGrpc.RoomCategoriesServiceImplBase(), Closeable {
    override fun listRoomCategories(
        request: ListRoomCategoriesRequest,
        responseObserver: StreamObserver<ListRoomCategoriesResponse>
    ) {
        entityFactory.wrapGrpc(request, responseObserver)
        {
            ListRoomCategoriesResponse.newBuilder()
                .addAllRooms(
                    service.listRoomCategories(
                        request.hotelId,
                        request.includeDisabled,
                        request.includeHidden,
                    )
                ).build()
        }
    }

    override fun getRoomCategory(
        request: GetRoomCategoryRequest,
        responseObserver: StreamObserver<GetRoomCategoryResponse>
    ) {
        entityFactory.wrapGrpc(request, responseObserver) {
            GetRoomCategoryResponse.newBuilder()
                .setRoom(service.getRoomCategory(request.hotelId, request.roomCategoryId)).build()
        }
    }

    override fun addRoomCategory(
        request: AddRoomCategoryRequest,
        responseObserver: StreamObserver<AddRoomCategoryResponse>
    ) {
        entityFactory.wrapGrpc(request, responseObserver) {
            Preconditions.checkArgument(request.room.id == 0L, "Unable to assign id to a new category")
            AddRoomCategoryResponse.newBuilder()
                .setRoom(
                    service.createRoomCategory(
                        request.hotelId,
                        request.room.name,
                        request.room.description,
                        request.room.type,
                        request.room.maxPrimaryPlaces,
                        request.room.maxExtraPlaces,
                        request.createDefaultPlacements
                    )
                )
                .build()
        }
    }

    override fun deleteRoomCategory(
        request: DeleteRoomCategoryRequest,
        responseObserver: StreamObserver<DeleteRoomCategoryResponse>
    ) {
        entityFactory.wrapGrpc(request, responseObserver) {
            service.deleteRoomCategory(request.hotelId, request.roomCategoryId)
            DeleteRoomCategoryResponse.getDefaultInstance()
        }
    }

    override fun hideRoomCategory(
        request: HideRoomCategoryRequest,
        responseObserver: StreamObserver<HideRoomCategoryResponse>
    ) {
        entityFactory.wrapGrpc(request, responseObserver) {
            service.hideRoomCategory(request.hotelId, request.roomCategoryId)
            HideRoomCategoryResponse.getDefaultInstance()
        }
    }

    override fun showRoomCategory(
        request: ShowRoomCategoryRequest,
        responseObserver: StreamObserver<ShowRoomCategoryResponse>
    ) {
        entityFactory.wrapGrpc(request, responseObserver) {
            service.showRoomCategory(request.hotelId, request.roomCategoryId)
            ShowRoomCategoryResponse.getDefaultInstance()
        }
    }

    override fun disableRoomCategory(
        request: DisableRoomCategoryRequest,
        responseObserver: StreamObserver<DisableRoomCategoryResponse>
    ) {
        entityFactory.wrapGrpc(request, responseObserver) {
            service.disableRoomCategory(request.hotelId, request.roomCategoryId)
            DisableRoomCategoryResponse.getDefaultInstance()
        }
    }

    override fun enableRoomCategory(
        request: EnableRoomCategoryRequest,
        responseObserver: StreamObserver<EnableRoomCategoryResponse>
    ) {
        entityFactory.wrapGrpc(request, responseObserver) {
            service.enableRoomCategory(request.hotelId, request.roomCategoryId)
            EnableRoomCategoryResponse.getDefaultInstance()
        }
    }

    override fun close() {
        entityFactory.close()
    }
}
