package ru.yandex.travel.hotels.extranet.grpc

import io.grpc.Status
import io.grpc.StatusException
import io.grpc.stub.StreamObserver
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.orm.jpa.JpaTransactionManager
import org.springframework.transaction.support.TransactionTemplate
import ru.yandex.travel.hotels.extranet.errors.AuthorizationException
import ru.yandex.travel.hotels.extranet.errors.IllegalOperationException
import javax.persistence.EntityManagerFactory
import javax.persistence.EntityNotFoundException
import javax.validation.ConstraintViolationException

fun <TReq, TRsp> EntityManagerFactory.wrapGrpc(
    req: TReq,
    responseObserver: StreamObserver<TRsp>,
    function: (TReq) -> TRsp
) {
    val transactionTemplate = TransactionTemplate(JpaTransactionManager(this))
    transactionTemplate.execute {
        grpcWrapper(req, responseObserver, function)
    }
}

fun <TReq, TRsp> grpcWrapper(
    request: TReq,
    responseObserver: StreamObserver<TRsp>,
    function: (TReq) -> TRsp
) {
    try {
        val rv = function(request)
        responseObserver.onNext(rv)
        responseObserver.onCompleted()
    } catch (e: Throwable) {
        when (e) {
            is IllegalArgumentException, is ConstraintViolationException -> responseObserver.onError(
                Status.INVALID_ARGUMENT.withDescription(e.message).asException()
            )
            is EntityNotFoundException, is EmptyResultDataAccessException -> responseObserver.onError(
                Status.NOT_FOUND
                    .withDescription(e.message).asException()
            )
            is IllegalOperationException -> responseObserver.onError(
                Status.FAILED_PRECONDITION.withDescription(e.message).asException()
            )
            is AuthorizationException -> responseObserver.onError(
                Status.PERMISSION_DENIED.withDescription(e.message).asException()
            )
            is StatusException -> responseObserver.onError(e)
            else -> {
                responseObserver.onError(Status.INTERNAL.withCause(e).withDescription(e.message).asException())
            }
        }
        throw e
    }
}
