package ru.yandex.travel.hotels.extranet.service

import org.springframework.data.domain.PageRequest
import org.springframework.data.domain.Pageable
import org.springframework.data.domain.Sort
import ru.yandex.travel.commons.proto.ECurrency
import ru.yandex.travel.commons.proto.ProtoUtils
import ru.yandex.travel.commons.proto.TPrice
import ru.yandex.travel.hotels.extranet.TDateIntervalFilter
import ru.yandex.travel.hotels.extranet.TSortField
import ru.yandex.travel.orders.commons.proto.TPage
import java.math.BigDecimal
import java.time.Instant
import java.time.LocalDate
import java.time.ZoneOffset
import javax.persistence.criteria.CriteriaBuilder
import javax.persistence.criteria.Path
import javax.persistence.criteria.Predicate

fun TPage.toPageable(proto: List<TSortField>, defaultSort: Sort): Pageable {
    val sort = if (proto.isEmpty()) {
        defaultSort
    } else {
        proto.fold(Sort.unsorted()) { acc, next ->
            acc.and(
                Sort.by(
                    Sort.Direction.valueOf(
                        next.direction.name.substring(3)
                    ), next.field
                )
            )
        }
    }
    return PageRequest.of(
        this.num,
        this.size,
        sort
    )
}

fun BigDecimal?.toTPrice(): TPrice {
    if (this == null) {
        return TPrice.getDefaultInstance()
    }
    return TPrice.newBuilder()
        .setCurrency(ECurrency.C_RUB)
        .setPrecision(2)
        .setAmount(this.multiply(BigDecimal.valueOf(100)).longValueExact())
        .build()
}

fun betweenPredicateDate(
    builder: CriteriaBuilder,
    property: Path<LocalDate>,
    dateFilter: TDateIntervalFilter
): Predicate {
    builder.between(
        property,
        ProtoUtils.toLocalDate(dateFilter.from),
        if (dateFilter.hasTill()) ProtoUtils.toLocalDate(dateFilter.till)
        else LocalDate.now(),
    )
    val predicateFrom: Predicate? =
        if (dateFilter.hasFrom()) builder.greaterThanOrEqualTo(property, ProtoUtils.toLocalDate(dateFilter.from)) else null
    val predicateTill: Predicate? =
        if (dateFilter.hasTill()) builder.lessThanOrEqualTo(property, ProtoUtils.toLocalDate(dateFilter.till)) else null
    return if (predicateFrom != null && predicateTill != null)
        builder.and(predicateFrom, predicateTill)
    else (predicateFrom ?: predicateTill)!!
}

fun betweenPredicateDateTime(
    builder: CriteriaBuilder,
    property: Path<Instant>,
    dateFilter: TDateIntervalFilter
): Predicate = builder.between(
    property,
    ProtoUtils.toLocalDate(dateFilter.from).atStartOfDay()
        .toInstant(ZoneOffset.UTC),
    if (dateFilter.hasTill())
        ProtoUtils.toLocalDate(dateFilter.till)
            .plusDays(1)
            .atStartOfDay()
            .toInstant(ZoneOffset.UTC)
    else Instant.now(),
)
