package ru.yandex.travel.hotels.extranet.service.bankorders

import io.grpc.StatusException
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.Sort
import org.springframework.data.jpa.domain.Specification
import org.springframework.stereotype.Service
import ru.yandex.travel.commons.proto.ProtoUtils
import ru.yandex.travel.hotels.extranet.EBankOrderTransactionPaymentType.BTPT_COST
import ru.yandex.travel.hotels.extranet.EBankOrderTransactionPaymentType.BTPT_YANDEX_ACCOUNT_COST_WITHDRAW
import ru.yandex.travel.hotels.extranet.TBankOrderInfo
import ru.yandex.travel.hotels.extranet.TGetBankOrderDetailsReq
import ru.yandex.travel.hotels.extranet.TGetBankOrderDetailsRsp
import ru.yandex.travel.hotels.extranet.TGetBankOrderInfoDetail
import ru.yandex.travel.hotels.extranet.TGetBankOrderInfoReq
import ru.yandex.travel.hotels.extranet.TGetBankOrderInfoRsp
import ru.yandex.travel.hotels.extranet.TGuest
import ru.yandex.travel.hotels.extranet.TPriceBreakdown
import ru.yandex.travel.hotels.extranet.entities.HotelAgreement
import ru.yandex.travel.hotels.extranet.entities.HotelIdentifier
import ru.yandex.travel.hotels.extranet.entities.Permission
import ru.yandex.travel.hotels.extranet.entities.orders.BankOrder
import ru.yandex.travel.hotels.extranet.entities.orders.BankOrderDetail
import ru.yandex.travel.hotels.extranet.entities.orders.BankOrderDetailWebView
import ru.yandex.travel.hotels.extranet.entities.orders.FETCHED_STATUS
import ru.yandex.travel.hotels.extranet.repository.BankOrderDetailsRepository
import ru.yandex.travel.hotels.extranet.repository.BankOrderDetailsWebViewRepository
import ru.yandex.travel.hotels.extranet.repository.BankOrdersRepository
import ru.yandex.travel.hotels.extranet.service.betweenPredicateDate
import ru.yandex.travel.hotels.extranet.service.hotels.HotelInfoService
import ru.yandex.travel.hotels.extranet.service.roles.UserRoleService
import ru.yandex.travel.hotels.extranet.service.toPageable
import ru.yandex.travel.hotels.extranet.service.toTPrice
import java.math.BigDecimal
import java.util.EnumSet
import javax.persistence.criteria.CriteriaBuilder
import javax.persistence.criteria.CriteriaQuery
import javax.persistence.criteria.Predicate
import javax.persistence.criteria.Root
import javax.transaction.Transactional

@Service
@Transactional
open class BankOrdersServiceImpl @Autowired constructor(
    private val repo: BankOrdersRepository,
    private val detailsRepo: BankOrderDetailsRepository,
    private val webViewRepo: BankOrderDetailsWebViewRepository,
    private val userRoleService: UserRoleService,
    private val hotelInfoService: HotelInfoService,
) : BankOrdersService {

    private val log = LoggerFactory.getLogger(javaClass)

    @Throws(StatusException::class)
    override fun getBankOrderDetails(req: TGetBankOrderDetailsReq): TGetBankOrderDetailsRsp {
        repo.findHotelsForPaymentBatchId(req.paymentBatchId).forEach {
            userRoleService.checkPermission(Permission.VIEW_PAYMENTS, hotelPartnerId = it)
        }
        val rsp = TGetBankOrderDetailsRsp.newBuilder()

        val detailsView = webViewRepo.findAllByIdPaymentBatchId(req.paymentBatchId)
        if (detailsView.isEmpty()) {
            fallbackBankOrderDetails(req.paymentBatchId, rsp)
        } else {
            val calculatedSum = detailsView.sumOf { it.signedPaidAmount ?: BigDecimal.ZERO }
            val bankOrderSum = repo.findById(req.paymentBatchId).orElseThrow().sum

            if (bankOrderSum.compareTo(calculatedSum) != 0) {
                log.error(
                    "Inconsistent data for payment batch id ${req.paymentBatchId}: " +
                        "$calculatedSum vs $bankOrderSum"
                )
                fallbackBankOrderDetails(req.paymentBatchId, rsp)
            } else {
                detailsView.forEach {
                    rsp.addResult(toProto(it))
                }
            }
        }
        return rsp.build()
    }

    private fun fallbackBankOrderDetails(paymentBatchId: String, rsp: TGetBankOrderDetailsRsp.Builder) {
        log.warn("Couldn't find calculated view for paymentBatchId ${paymentBatchId}, using the fallback")
        detailsRepo.findAllByBankOrderPaymentBatchId(paymentBatchId)
            .filter {
                it.sum != BigDecimal.ZERO && EnumSet.of(BTPT_YANDEX_ACCOUNT_COST_WITHDRAW, BTPT_COST)
                    .contains(it.paymentType)
            }.map {
                val proto = TGetBankOrderInfoDetail.newBuilder()
                proto.transactionType = it.transactionType
                proto.priceBreakdown = TPriceBreakdown.newBuilder()
                    .setPartner(it.signedSum.toTPrice())
                    .build()
                proto
            }.forEach {
                rsp.addResult(it)
            }
    }

    private fun toProto(detail: BankOrderDetailWebView): TGetBankOrderInfoDetail.Builder {
        val proto = TGetBankOrderInfoDetail.newBuilder()
        proto.transactionType = detail.id.transactionType
        detail.hotelId?.let { proto.hotel = hotelInfoService.getHotelInfo(it) }
        proto.orderId = detail.id.orderId
        proto.orderPrettyId = detail.prettyId
        detail.guestFirstName?.let {
            val guest = TGuest.newBuilder()
            guest.firstName = detail.guestFirstName
            guest.lastName = detail.guestLastName!!
            guest.build()
        }?.let {
            proto.firstGuest = it
        }
        proto.orderCreatedAt = ProtoUtils.fromInstant(detail.orderCreatedAt)
        proto.checkInDate = ProtoUtils.toTDate(detail.checkInDate)
        proto.checkOutDate = ProtoUtils.toTDate(detail.checkOutDate)
        proto.priceBreakdown = TPriceBreakdown.newBuilder()
            .setFiscalPrice(detail.signedFiscalPrice.toTPrice())
            .setHotelPrice(detail.signedHotelPrice.toTPrice())
            .setPartner(detail.signedPaidAmount.toTPrice())
            .build()
        return proto
    }

    @Throws(StatusException::class)
    override fun getBankOrders(req: TGetBankOrderInfoReq): TGetBankOrderInfoRsp {
        val rsp = TGetBankOrderInfoRsp.newBuilder()
        val hotelId = HotelIdentifier.fromProto(req.hotelId)
        rsp.hotelInfo = hotelInfoService.getHotelInfo(hotelId)
        userRoleService.checkPermission(Permission.VIEW_PAYMENTS, hotelPartnerId = hotelId)
        val result = repo.findAll(
            GetInfoSpecImpl(req), req.page.toPageable(
                req.sortList,
                Sort.by(Sort.Direction.DESC, "paymentBatchId")
            )
        )

        rsp.totalRecords = result.totalElements
        result.forEach { bankOrder ->
            this.toProto(bankOrder).let { proto -> rsp.addResult(proto) }
        }

        return rsp.build()
    }

    private fun toProto(entity: BankOrder): TBankOrderInfo {
        val builder = TBankOrderInfo.newBuilder()
        builder.bankOrderId = entity.bankOrderId
        builder.paymentBatchId = entity.paymentBatchId
        builder.status = entity.status
        builder.oebsStatus = entity.oebsStatus
        builder.sum = entity.sum.toString()
        builder.description = entity.description
        builder.updatedAt = ProtoUtils.fromInstant(entity.updatedAtInOrc)
        builder.eventTime = ProtoUtils.toTDate(entity.eventTime)
        return builder.build()
    }

    private class GetInfoSpecImpl constructor(val proto: TGetBankOrderInfoReq) : Specification<BankOrder> {
        /**
         * The approximate query being generated (with no optional filters and pagination)
         * <code>
         *     SELECT * FROM bank_orders WHERE payment_batch_id IN (
         *       SELECT payment_batch_id FROM bank_order_details
         *       WHERE contract_id IN (
         *         SELECT financial_contract_id FROM hotel_agreements
         *         WHERE hotel_id = ? AND partner_id = ?
         *       ) subquery1
         *     ) subquery2
         *     AND fetch_status = 'fetched'
         * </code>
         */
        override fun toPredicate(
            bankOrder: Root<BankOrder>,
            query: CriteriaQuery<*>,
            builder: CriteriaBuilder
        ): Predicate {

            val subquery1 = query.subquery(Long::class.java)
            val hotelAgreement = subquery1.from(HotelAgreement::class.java)
            subquery1.select(hotelAgreement.get("financialContractId")).where(
                builder.equal(
                    hotelAgreement.get<HotelIdentifier>("hotelIdentifier"),
                    HotelIdentifier.fromProto(proto.hotelId)
                )
            )

            val subquery2 = query.subquery(BankOrder::class.java)
            val bankOrderDetail = subquery2.from(BankOrderDetail::class.java)
            subquery2.select(bankOrderDetail.get("bankOrder")).where(
                builder.`in`(bankOrderDetail.get<Long>("contractId")).value(subquery1)
            ).distinct(true)

            val mainQuery = builder.and(
                bankOrder.`in`(subquery2),
                builder.equal(bankOrder.get<String>("fetchStatus"), FETCHED_STATUS)
            )
            return if (proto.hasEventTime()) {
                builder.and(
                    mainQuery,
                    betweenPredicateDate(builder, bankOrder.get("eventTime"), proto.eventTime)
                )
            } else mainQuery
        }
    }
}
