package ru.yandex.travel.hotels.extranet.service.blackbox

import com.fasterxml.jackson.databind.ObjectMapper
import lombok.extern.slf4j.Slf4j
import org.asynchttpclient.RequestBuilder
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Service
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper
import ru.yandex.travel.tvm.TvmWrapper
import java.io.IOException
import java.net.InetAddress

@Service
@Slf4j
@EnableConfigurationProperties(BlackBoxServiceProperties::class)
class BlackBoxServiceImpl @Autowired constructor(
    @Qualifier("blackboxAsyncHttpClientWrapper")
    val ahcWrapper: AsyncHttpClientWrapper,
    val tvm: TvmWrapper?,
    val config: BlackBoxServiceProperties,
    val objectMapper: ObjectMapper,
) : BlackBoxService {
    companion object {
        val LOG = LoggerFactory.getLogger(BlackBoxServiceImpl::class.java)
        val SERVICE_TICKET_HEADER_NAME = "X-Ya-Service-Ticket"
        val SUCCESSFUL_RESPONSE_CODES = setOf(200, 201)
    }


    override fun getUidByLogin(login: String, userIp: String?): Long? {
        val req = RequestBuilder("GET")
            .setUrl(config.url)
            .addQueryParam("method", "userinfo")
            .addQueryParam("login", login)
            .addQueryParam("userip", userIp ?: InetAddress.getLocalHost().hostAddress)
            .addQueryParam("format", "json")
        tvm?.getServiceTicket(config.tvmAlias)?.apply { req.setHeader(SERVICE_TICKET_HEADER_NAME, this) }
        val response = ahcWrapper.executeRequest(req).join()
        if (response.statusCode !in SUCCESSFUL_RESPONSE_CODES) {
            LOG.warn("Unexpected blackbox response status: ${response.statusCode}")
            return null
        }
        try {
            val bbResp = objectMapper.readValue(response.responseBody, BlackBoxResponse::class.java)
            return bbResp.users?.first()?.id
        } catch (t: IOException) {
            LOG.warn("Error while parsing blackbox response", t)
            return null
        }
    }
}
