package ru.yandex.travel.hotels.extranet.service.content.annulation

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.hotels.extranet.dto.AnnulationPolicyDTO
import ru.yandex.travel.hotels.extranet.dto.AnnulationRuleDTO
import ru.yandex.travel.hotels.extranet.entities.AnnulationPenaltyType
import ru.yandex.travel.hotels.extranet.entities.AnnulationPolicy
import ru.yandex.travel.hotels.extranet.entities.AnnulationRule
import ru.yandex.travel.hotels.extranet.entities.Organization
import ru.yandex.travel.hotels.extranet.entities.Permission
import ru.yandex.travel.hotels.extranet.repository.AnnulationPolicyRepository
import ru.yandex.travel.hotels.extranet.repository.OrganizationRepository
import ru.yandex.travel.hotels.extranet.service.roles.UserRoleService
import java.math.BigDecimal
import java.util.UUID
import javax.persistence.EntityNotFoundException
import javax.transaction.Transactional

@Service
@Transactional
class AnnulationPoliciesServiceImpl @Autowired constructor(
    private val organizationRepository: OrganizationRepository,
    private val policyRepository: AnnulationPolicyRepository,
    private val userRoleService: UserRoleService,
    private val annulationPolicyMapper: AnnulationPolicyMapper
) : AnnulationPoliciesService {

    override fun list(organizationId: UUID): List<AnnulationPolicyDTO> {
        userRoleService.checkPermission(Permission.MANAGE_ANNULATION_POLICIES, organizationId = organizationId)
        return policyRepository.findAllByOrganizationIdAndDeletedFalse(organizationId).map { annulationPolicyMapper.mapToDto(it) }
    }

    override fun create(organizationId: UUID, name: String, rules: List<AnnulationRuleDTO>): AnnulationPolicyDTO {
        userRoleService.checkPermission(Permission.MANAGE_ANNULATION_POLICIES, organizationId = organizationId)
        val org = organizationRepository.findById(organizationId).orElseThrow { EntityNotFoundException() }
        val policy = AnnulationPolicy(org, name, rules.map { annulationPolicyMapper.mapRuleFromDto(it) })
        val res = policyRepository.saveAndFlush(policy)
        return annulationPolicyMapper.mapToDto(res)
    }

    override fun update(
        organizationId: UUID,
        policyId: Long,
        name: String?,
        rules: List<AnnulationRuleDTO>?
    ): AnnulationPolicyDTO {
        userRoleService.checkPermission(Permission.MANAGE_ANNULATION_POLICIES, organizationId = organizationId)
        val policy =
            policyRepository.getByOrganizationIdAndIdAndDeletedIsFalse(organizationId, policyId)
        name?.let { policy.name = it }
        rules?.let { r ->
            policy.rules.clear()
            r.map { annulationPolicyMapper.mapRuleFromDto(it) }.forEach { policy.addRule(it) }
            policyRepository.flush()
        }
        return annulationPolicyMapper.mapToDto(policy)
    }

    override fun delete(organizationId: UUID, policyId: Long) {
        userRoleService.checkPermission(Permission.MANAGE_ANNULATION_POLICIES, organizationId = organizationId)
        val policy =
            policyRepository.getByOrganizationIdAndIdAndDeletedIsFalse(organizationId, policyId)
        policy.deleted = true
    }

    override fun createDefault(organizationId: UUID) {
        userRoleService.checkPermission(Permission.MANAGE_ANNULATION_POLICIES, organizationId = organizationId)
        val org = organizationRepository.getOne(organizationId)
        policyRepository.saveAll(
            listOf(
                createNonRef(org),
                createFullyRef(org),
                createFirstNightRef(org)
            )
        )
    }

    private fun createNonRef(org: Organization): AnnulationPolicy {
        return AnnulationPolicy(
            org, "Без возможности аннуляции",
            listOf(AnnulationRule(AnnulationPenaltyType.FULL))
        )
    }

    private fun createFullyRef(org: Organization): AnnulationPolicy {
        return AnnulationPolicy(
            org, "C полной аннуляцией",
            listOf(AnnulationRule(AnnulationPenaltyType.NONE))
        )
    }

    private fun createFirstNightRef(org: Organization): AnnulationPolicy {
        return AnnulationPolicy(
            org, "Штраф за аннуляцию — первая ночь", listOf(
                AnnulationRule(AnnulationPenaltyType.NIGHTS, penaltyNominal = BigDecimal.valueOf(1))
            )
        )
    }
}
