package ru.yandex.travel.hotels.extranet.service.content.annulation

import org.springframework.stereotype.Component
import ru.yandex.travel.hotels.extranet.dto.AnnulationPolicyDTO
import ru.yandex.travel.hotels.extranet.dto.AnnulationPolicyMoment
import ru.yandex.travel.hotels.extranet.dto.AnnulationRuleDTO
import ru.yandex.travel.hotels.extranet.dto.PenaltyType
import ru.yandex.travel.hotels.extranet.dto.TimeUnit
import ru.yandex.travel.hotels.extranet.entities.AnnulationPenaltyType
import ru.yandex.travel.hotels.extranet.entities.AnnulationPolicy
import ru.yandex.travel.hotels.extranet.entities.AnnulationRule
import ru.yandex.travel.hotels.extranet.entities.AnnulationRuleStart
import java.math.BigDecimal

@Component
class AnnulationPolicyMapper {
    fun mapToDto(policy: AnnulationPolicy): AnnulationPolicyDTO {
        val mappedRules = policy.rules.sortedBy { it.comparator() }.reversed().map { mapRule(it) }
        val ruleDTOsWithEndsAt = (mappedRules + mappedRules.last()).zipWithNext { a, b ->
            if (b.hasStartsAt() && a != b) {
                a.toBuilder().setEndsAt(b.startsAt).build()
            } else {
                a
            }
        }

        return AnnulationPolicyDTO.newBuilder()
            .setId(policy.id ?: 0)
            .setName(policy.name)
            .setDescription(policy.describe())
            .addAllRules(ruleDTOsWithEndsAt)
            .build()
    }

    fun mapRuleFromDto(dto: AnnulationRuleDTO): AnnulationRule {
        require(!dto.hasEndsAt()) { "rules should not specify ends_at explicitly" }
        return AnnulationRule(
            mapTypeFromDto(dto.penaltyType),
            start = if (dto.hasStartsAt()) {
                AnnulationRuleStart(
                    dto.startsAt.amount,
                    when (dto.startsAt.units) {
                        TimeUnit.TIME_UNIT_DAY -> AnnulationRuleStart.Unit.DAYS
                        TimeUnit.TIME_UNIT_HOUR -> AnnulationRuleStart.Unit.HOURS
                        else -> throw IllegalArgumentException("unsupported time unit")
                    }
                )
            } else {
                null
            },
            penaltyNominal = when (dto.penaltyAmountCase) {
                AnnulationRuleDTO.PenaltyAmountCase.PENALTY_AMOUNT_FLOAT -> BigDecimal.valueOf(dto.penaltyAmountFloat.toDouble())
                AnnulationRuleDTO.PenaltyAmountCase.PENALTY_AMOUNT_INT -> BigDecimal.valueOf(dto.penaltyAmountInt.toLong())
                AnnulationRuleDTO.PenaltyAmountCase.PENALTYAMOUNT_NOT_SET, null -> null
            }
        )
    }

    private fun mapRule(rule: AnnulationRule): AnnulationRuleDTO {
        val builder = AnnulationRuleDTO.newBuilder()
            .setPenaltyType(mapTypeToDto(rule.penaltyType))
        rule.penaltyNominal?.let {
            when (rule.penaltyType) {
                AnnulationPenaltyType.NIGHTS -> builder.setPenaltyAmountInt(it.intValueExact())
                AnnulationPenaltyType.FIXED, AnnulationPenaltyType.PERCENTAGE -> builder.setPenaltyAmountFloat(it.toFloat())
                else -> {}
            }
        }
        rule.start?.let {
            builder.setStartsAt(
                AnnulationPolicyMoment.newBuilder()
                    .setAmount(it.amount)
                    .setUnits(
                        when (it.unit) {
                            AnnulationRuleStart.Unit.HOURS -> TimeUnit.TIME_UNIT_HOUR
                            AnnulationRuleStart.Unit.DAYS -> TimeUnit.TIME_UNIT_DAY
                        }
                    )
                    .build()
            )
        }
        return builder.build()
    }

    private fun mapTypeFromDto(penaltyTypeDto: PenaltyType): AnnulationPenaltyType {
        return when (penaltyTypeDto) {
            PenaltyType.PENALTY_TYPE_NONE -> AnnulationPenaltyType.NONE
            PenaltyType.PENALTY_TYPE_FIXED -> AnnulationPenaltyType.FIXED
            PenaltyType.PENALTY_TYPE_PERCENTAGE -> AnnulationPenaltyType.PERCENTAGE
            PenaltyType.PENALTY_TYPE_FULL -> AnnulationPenaltyType.FULL
            PenaltyType.PENALTY_TYPE_NIGHTS -> AnnulationPenaltyType.NIGHTS
            else -> {
                throw IllegalArgumentException("unsupported penalty type")
            }
        }
    }

    private fun mapTypeToDto(penaltyType: AnnulationPenaltyType): PenaltyType {
        return when (penaltyType) {
            AnnulationPenaltyType.NONE -> PenaltyType.PENALTY_TYPE_NONE
            AnnulationPenaltyType.FIXED -> PenaltyType.PENALTY_TYPE_FIXED
            AnnulationPenaltyType.PERCENTAGE -> PenaltyType.PENALTY_TYPE_PERCENTAGE
            AnnulationPenaltyType.FULL -> PenaltyType.PENALTY_TYPE_FULL
            AnnulationPenaltyType.NIGHTS -> PenaltyType.PENALTY_TYPE_NIGHTS
        }
    }
}

