package ru.yandex.travel.hotels.extranet.service.content.rateplans

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.hotels.extranet.dto.RatePlanDTO
import ru.yandex.travel.hotels.extranet.entities.Permission
import ru.yandex.travel.hotels.extranet.entities.RatePlan
import ru.yandex.travel.hotels.extranet.entities.RoomCategory
import ru.yandex.travel.hotels.extranet.repository.AnnulationPolicyRepository
import ru.yandex.travel.hotels.extranet.repository.HotelRepository
import ru.yandex.travel.hotels.extranet.repository.RatePlanRepository
import ru.yandex.travel.hotels.extranet.repository.RoomCategoryRepository
import ru.yandex.travel.hotels.extranet.service.roles.UserRoleService
import javax.persistence.EntityNotFoundException
import javax.transaction.Transactional

@Service
@Transactional
class RatePlanServiceImpl @Autowired constructor(
    private val ratePlanRepository: RatePlanRepository,
    private val hotelRepository: HotelRepository,
    private val annulationPolicyRepository: AnnulationPolicyRepository,
    private val roomCategoryRepository: RoomCategoryRepository,
    private val ratePlanMapper: RatePlanMapper,
    private val userRoleService: UserRoleService,
) : RatePlanService {

    override fun listRatePlans(hotelId: Long, includeDisabled: Boolean, includeHidden: Boolean): List<RatePlanDTO> {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        var res = ratePlanRepository.findAllByHotelIdAndDeletedFalse(hotelId)
        if (!includeDisabled) {
            res = res.filter { !it.disabled }
        }
        if (!includeHidden) {
            res = res.filter { !it.hidden }
        }
        return res.map { ratePlanMapper.mapRatePlanToDto(it) }
    }

    override fun addRatePlan(
        hotelId: Long,
        name: String,
        annulationPolicyId: Long,
        description: String?,
        roomCategoryIds: Set<Long>
    ): RatePlanDTO {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        val hotel = hotelRepository.getOne(hotelId)
        val policy = annulationPolicyRepository.getByOrganizationIdAndIdAndDeletedIsFalse(
            hotel.organization.id,
            annulationPolicyId
        )
        val categories = getCategories(hotelId, roomCategoryIds)
        val plan = RatePlan(hotel, policy, name, description, categories)
        return ratePlanMapper.mapRatePlanToDto(ratePlanRepository.saveAndFlush(plan))
    }

    override fun deleteRatePlan(hotelId: Long, id: Long) {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        val plan = ratePlanRepository.findByHotelIdAndIdAndDeletedFalse(hotelId, id)
        plan.deleted = true
    }

    override fun updateRatePlan(
        hotelId: Long,
        id: Long,
        name: String?,
        annulationPolicyId: Long?,
        description: String?,
        roomCategories: Set<Long>?
    ): RatePlanDTO {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        val plan = ratePlanRepository.findByHotelIdAndIdAndDeletedFalse(hotelId, id)
        name?.let { plan.name = it }
        annulationPolicyId?.let {
            plan.annulationPolicy = annulationPolicyRepository.getByOrganizationIdAndIdAndDeletedIsFalse(
                plan.hotel.organization.id,
                it
            )
        }
        description?.let {
            plan.description = it.ifEmpty { null }
        }
        roomCategories?.let { plan.roomCategories = getCategories(hotelId, it) }
        return ratePlanMapper.mapRatePlanToDto(plan)
    }

    override fun getRatePlan(hotelId: Long, id: Long): RatePlanDTO {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        val plan = ratePlanRepository.findByHotelIdAndIdAndDeletedFalse(hotelId, id)
        return ratePlanMapper.mapRatePlanToDto(plan)
    }

    override fun hideRatePlan(hotelId: Long, id: Long) {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        val plan = ratePlanRepository.findByHotelIdAndIdAndDeletedFalse(hotelId, id)
        plan.hidden = true
    }

    override fun showRatePlan(hotelId: Long, id: Long) {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        val plan = ratePlanRepository.findByHotelIdAndIdAndDeletedFalse(hotelId, id)
        plan.hidden = false
    }

    override fun disableRatePlan(hotelId: Long, id: Long) {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        val plan = ratePlanRepository.findByHotelIdAndIdAndDeletedFalse(hotelId, id)
        plan.disabled = true
    }

    override fun enableRatePlan(hotelId: Long, id: Long) {
        userRoleService.checkPermission(Permission.MANAGE_RATE_PLANS, hotelId = hotelId)
        val plan = ratePlanRepository.findByHotelIdAndIdAndDeletedFalse(hotelId, id)
        plan.disabled = false
    }

    private fun getCategories(
        hotelId: Long,
        roomCategoryIds: Set<Long>
    ): Set<RoomCategory> {
        val categories = roomCategoryRepository.findAllByHotelIdAndIdInAndDeletedIsFalse(hotelId, roomCategoryIds)
        if (categories.size != roomCategoryIds.size) {
            val missingCategories =
                roomCategoryIds.filter { catId -> !categories.map { it.id }.toSet().contains(catId) }
            throw EntityNotFoundException("Category with id(s) $missingCategories is (are) not found")
        }
        return categories
    }
}
