package ru.yandex.travel.hotels.extranet.service.content.roomcat

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.hotels.extranet.dto.RoomCategoryDTO
import ru.yandex.travel.hotels.extranet.dto.RoomPlacementDTO
import ru.yandex.travel.hotels.extranet.entities.Permission
import ru.yandex.travel.hotels.extranet.entities.RoomCategory
import ru.yandex.travel.hotels.extranet.entities.RoomCategoryType
import ru.yandex.travel.hotels.extranet.entities.RoomPlacement
import ru.yandex.travel.hotels.extranet.repository.HotelRepository
import ru.yandex.travel.hotels.extranet.repository.RoomCategoryRepository
import ru.yandex.travel.hotels.extranet.service.roles.UserRoleService
import javax.transaction.Transactional
import ru.yandex.travel.hotels.extranet.dto.RoomCategoryType as RoomCategoryTypeDTO

@Service
@Transactional
class RoomCategoryServiceImpl @Autowired constructor(
    private val roomCategoryRepository: RoomCategoryRepository,
    private val userRoleService: UserRoleService,
    private val hotelRepository: HotelRepository,
    private val roomCategoryMapper: RoomCategoryMapper,
) : RoomCategoryService {

    override fun createRoomCategory(
        hotelId: Long,
        name: String,
        description: String,
        type: RoomCategoryTypeDTO,
        maxPrimary: Int,
        maxExtra: Int,
        createDefaultPlacements: Boolean
    ): RoomCategoryDTO {
        checkPermission(hotelId)
        val hotel = hotelRepository.getOne(hotelId)
        val roomCategory = RoomCategory(
            hotel, name, RoomCategoryType.fromProto(type), maxPrimary, maxExtra
        )
        roomCategory.description = description
        if (createDefaultPlacements) {
            roomCategory.placements.addAll(createDefaultPlacements(roomCategory))
        }
        return roomCategoryMapper.mapRoomCategory(roomCategoryRepository.saveAndFlush(roomCategory))
    }

    override fun getRoomCategory(hotelId: Long, id: Long): RoomCategoryDTO {
        checkPermission(hotelId)
        val roomCat = roomCategoryRepository.getRoomCategoryByHotelIdAndIdAndDeletedIsFalse(hotelId, id)
        return roomCategoryMapper.mapRoomCategory(roomCat)
    }

    override fun deleteRoomCategory(hotelId: Long, id: Long) {
        checkPermission(hotelId)
        val roomCat = roomCategoryRepository.getRoomCategoryByHotelIdAndIdAndDeletedIsFalse(hotelId, id)
        roomCat.deleted = true
    }

    override fun listRoomCategories(
        hotelId: Long,
        includeDisabled: Boolean,
        includeHidden: Boolean
    ): List<RoomCategoryDTO> {
        checkPermission(hotelId)
        var res = roomCategoryRepository.findAllByHotelIdAndDeletedIsFalse(hotelId)
        if (!includeDisabled) {
            res = res.filter { !it.disabled }
        }
        if (!includeHidden) {
            res = res.filter { !it.hidden }
        }
        return res.map { roomCategoryMapper.mapRoomCategory(it) }
    }

    override fun hideRoomCategory(hotelId: Long, id: Long) {
        checkPermission(hotelId)
        val roomCat = roomCategoryRepository.getRoomCategoryByHotelIdAndIdAndDeletedIsFalse(hotelId, id)
        roomCat.hidden = true
    }

    override fun showRoomCategory(hotelId: Long, id: Long) {
        checkPermission(hotelId)
        val roomCat = roomCategoryRepository.getRoomCategoryByHotelIdAndIdAndDeletedIsFalse(hotelId, id)
        roomCat.hidden = false
    }

    override fun disableRoomCategory(hotelId: Long, id: Long) {
        checkPermission(hotelId)
        val roomCat = roomCategoryRepository.getRoomCategoryByHotelIdAndIdAndDeletedIsFalse(hotelId, id)
        roomCat.disabled = true
    }

    override fun enableRoomCategory(hotelId: Long, id: Long) {
        checkPermission(hotelId)
        val roomCat = roomCategoryRepository.getRoomCategoryByHotelIdAndIdAndDeletedIsFalse(hotelId, id)
        roomCat.disabled = false
    }

    private fun checkPermission(hotelId: Long) {
        userRoleService.checkPermission(Permission.MANAGE_ROOM_TYPES, hotelId = hotelId)
    }

    private fun createDefaultPlacements(roomCategory: RoomCategory): List<RoomPlacement> {
        val placements = mutableListOf<RoomPlacement>()
        for (i in 1..roomCategory.maxPrimaryPlaces) {
            val prefix = when (i) {
                1 -> "Одноместное"
                2 -> "Двухместное"
                3 -> "Трехместное"
                else -> "$i-местное"
            }
            for (j in 0..roomCategory.maxExtraPlaces) {
                val extra = when (j) {
                    0 -> ""
                    1 -> " c одним доп. местом"
                    else -> " c $j доп. местами"
                }
                placements.add(RoomPlacement(roomCategory, "$prefix размещение$extra", i, j))
            }
        }
        return placements
    }
}
