package ru.yandex.travel.hotels.extranet.service.hotels

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.hotels.extranet.AuthorizedHotel
import ru.yandex.travel.hotels.extranet.cache.IHotelConnectionService
import ru.yandex.travel.hotels.extranet.entities.Hotel
import ru.yandex.travel.hotels.extranet.entities.HotelIdentifier
import ru.yandex.travel.hotels.extranet.entities.HotelManagementSource
import ru.yandex.travel.hotels.extranet.entities.Permission
import ru.yandex.travel.hotels.extranet.repository.HotelRepository
import ru.yandex.travel.hotels.extranet.service.roles.UserRoleService

@Service
class HotelInfoServiceImpl @Autowired constructor(
    private val hotelRepository: HotelRepository,
    private val hotelConnectionService: IHotelConnectionService,
    private val userRoleService: UserRoleService,

    ) : HotelInfoService {
    override fun getHotelInfo(id: HotelIdentifier): AuthorizedHotel {
        userRoleService.checkPermission(Permission.VIEW_HOTELS, hotelPartnerId = id)
        return getHotelInfo(hotelRepository.findHotelByPartnerHotelId(id)!!)
    }

    override fun listAvailableHotels(): List<AuthorizedHotel> {
        val user = userRoleService.getCurrentUser() ?: return emptyList()
        val specificHotels =
            user.roles.filter { it.role.permissions.contains(Permission.VIEW_HOTELS) && it.hotel != null }
                .map { it.hotel!! }.toSet()
        val orgLevelHotels = user.roles.filter {
            it.role.permissions.contains(Permission.VIEW_HOTELS)
                && it.hotel == null && it.organization != null
        }.flatMap { it.organization!!.hotels }.toSet()
        return (specificHotels + orgLevelHotels).map { getHotelInfo(it) }
    }

    private fun getHotelInfo(hotel: Hotel): AuthorizedHotel {
        if (hotel.managedBy == HotelManagementSource.YANDEX || hotel.partnerHotelId == null) {
            return AuthorizedHotel.newBuilder().apply {
                title = hotel.name
                if (hotel.permalink != null) {
                    permalink = hotel.permalink!!
                }
            }.build()
        } else {
            val proto = AuthorizedHotel.newBuilder()
            proto.hotelId = hotel.partnerHotelId!!.toProto()
            hotelConnectionService.getByHotelIdKey(hotel.partnerHotelId!!)?.let { h ->
                proto.title = h.getTitle()
                proto.permalink = h.permalink
            }
            return proto.build()
        }
    }
}
