package ru.yandex.travel.hotels.extranet.service.invitations

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.node.ObjectNode
import com.google.common.base.Preconditions
import com.google.common.base.Strings
import io.grpc.Status
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Service
import ru.yandex.travel.credentials.UserCredentials
import ru.yandex.travel.hotels.extranet.EApproveAuthStatus
import ru.yandex.travel.hotels.extranet.ERequestAuthStatus
import ru.yandex.travel.hotels.extranet.TApproveAuthReq
import ru.yandex.travel.hotels.extranet.TApproveAuthRsp
import ru.yandex.travel.hotels.extranet.TRequestAuthReq
import ru.yandex.travel.hotels.extranet.TRequestAuthRsp
import ru.yandex.travel.hotels.extranet.cache.IHotelConnectionService
import ru.yandex.travel.hotels.extranet.entities.Hotel
import ru.yandex.travel.hotels.extranet.entities.HotelIdentifier
import ru.yandex.travel.hotels.extranet.entities.HotelManagementSource
import ru.yandex.travel.hotels.extranet.entities.Invitation
import ru.yandex.travel.hotels.extranet.entities.Organization
import ru.yandex.travel.hotels.extranet.entities.Role
import ru.yandex.travel.hotels.extranet.entities.User
import ru.yandex.travel.hotels.extranet.entities.UserRoleBinding
import ru.yandex.travel.hotels.extranet.repository.HotelRepository
import ru.yandex.travel.hotels.extranet.repository.InvitationRepository
import ru.yandex.travel.hotels.extranet.repository.OrganizationRepository
import ru.yandex.travel.hotels.extranet.repository.UserRepository
import ru.yandex.travel.hotels.extranet.service.notification.MailSenderService
import ru.yandex.travel.hotels.proto.EPartnerId
import java.time.Instant
import java.util.UUID
import javax.transaction.Transactional

@Service
@EnableConfigurationProperties(InvitationServiceProperties::class)
open class InvitationServiceImpl @Autowired constructor(
    private val invitationRepository: InvitationRepository,
    private val hotelRepository: HotelRepository,
    private val userRepository: UserRepository,
    private val organizationRepository: OrganizationRepository,

    private val hotelConnectionService: IHotelConnectionService,
    private val mailSenderService: MailSenderService,

    private val invitationServiceProperties: InvitationServiceProperties,
) : InvitationService {

    override fun requestInvitation(request: TRequestAuthReq): TRequestAuthRsp {
        val userCredentials = UserCredentials.get()
        requestAuthorizationPreconditions(userCredentials, request)

        val hotelId = HotelIdentifier.fromProto(request.hotelId)
        if (invitationRepository.haveApprovedRequest(userCredentials.passportId, hotelId)) {
            return TRequestAuthRsp.newBuilder()
                .setStatus(ERequestAuthStatus.RAS_ALREADY_APPROVED)
                .build()
        }
        if (invitationRepository.haveActiveRequest(userCredentials.passportId, hotelId)) {
            return TRequestAuthRsp.newBuilder().setStatus(ERequestAuthStatus.RAS_ALREADY_REQUESTED).build()
        }
        val hotel = hotelConnectionService.getByHotelIdKey(hotelId, request.email)
            ?: return TRequestAuthRsp.newBuilder()
                .setStatus(ERequestAuthStatus.RAS_HOTEL_NOT_FOUND)
                .build()
        if (!invitationServiceProperties.allowAllFromYandexTeam || !request.email.lowercase()
                .endsWith("@yandex-team.ru")
        ) {
            val hotelEmails = hotel.getAllEmailsLowerCase()
            if (!hotelEmails.contains(request.email.lowercase())) {
                return TRequestAuthRsp.newBuilder()
                    .setStatus(ERequestAuthStatus.RAS_INVALID_EMAIL)
                    .build()
            }
        }

        val requestApproveToken = generateToken()
        sendInvitationEmail(userCredentials, request.email, requestApproveToken, hotel.getTitle())
        invitationRepository.saveAndFlush(
            Invitation(
                hotelIdentifier = hotelId,
                passportId = userCredentials.passportId,
                requestToken = requestApproveToken,
                requestExpiresAt = Instant.now().plus(invitationServiceProperties.invitationParams.expirationDuration),
            )
        )
        return TRequestAuthRsp.newBuilder()
            .setStatus(ERequestAuthStatus.RAS_OK)
            .build()
    }

    private fun requestAuthorizationPreconditions(userCredentials: UserCredentials, request: TRequestAuthReq) {
        try {
            Preconditions.checkArgument(
                !Strings.isNullOrEmpty(userCredentials.login),
                "userCredentials.login must not be null or empty"
            )
            Preconditions.checkArgument(
                !Strings.isNullOrEmpty(userCredentials.passportId),
                "userCredentials.passportId must not be null or empty"
            )
            Preconditions.checkArgument(
                !Strings.isNullOrEmpty(request.hotelId.originalId),
                "request.hotelId.originalId must not be null or empty"
            )
            Preconditions.checkArgument(
                request.hotelId.partnerId != null
                    && request.hotelId.partnerId != EPartnerId.PI_UNUSED
                    && request.hotelId.partnerId != EPartnerId.UNRECOGNIZED,
                "request.hotelId.partnerId must not be null or empty"
            )
            Preconditions.checkArgument(
                !Strings.isNullOrEmpty(request.email),
                "request.email must not be null or empty"
            )
        } catch (e: Throwable) {
            throw Status.INVALID_ARGUMENT.withCause(e).withDescription(e.message).asException()
        }
    }

    private fun generateToken(): String {
        return UUID.randomUUID().toString()
    }

    private fun sendInvitationEmail(
        userCredentials: UserCredentials,
        email: String,
        requestApproveToken: String,
        hotelTitle: String
    ) {
        val objectNode = ObjectMapper().valueToTree<ObjectNode>(
            AuthRequestEmailData(
                username = userCredentials.login,
                title = hotelTitle,
                token = requestApproveToken,
            )
        )
        mailSenderService.sendEmailSync(
            email,
            invitationServiceProperties.invitationParams.emailCampaign,
            objectNode,
            emptyList(),
        )
    }

    @Transactional
    override fun acceptInvitation(request: TApproveAuthReq): TApproveAuthRsp {
        val userCredentials = UserCredentials.get()
        acceptInvitationPrecondition(userCredentials, request)
        val authRequest = invitationRepository.getByRequestToken(request.token)
            ?: return TApproveAuthRsp.newBuilder()
                .setStatus(EApproveAuthStatus.AAS_INVALID_TOKEN)
                .build()
        if (authRequest.approvedAt != null) {
            return TApproveAuthRsp.newBuilder()
                .setStatus(EApproveAuthStatus.AAS_ALREADY_APPROVED)
                .build()
        }
        if (authRequest.requestExpiresAt < Instant.now()) {
            return TApproveAuthRsp.newBuilder()
                .setStatus(EApproveAuthStatus.AAS_EXPIRED_TOKEN)
                .build()
        }
        val user =
            userRepository.findById(userCredentials.parsedPassportIdOrNull).orElseGet { createUser(userCredentials) }
        val hotel =
            hotelRepository.findHotelByPartnerHotelId(authRequest.hotelIdentifier)
                ?: createHotelWithOrg(authRequest.hotelIdentifier)
        user.roles.add(
            UserRoleBinding(
                user,
                Role.MINOR_EXTRANET_VIEWER,
                organization = hotel.organization,
                hotel = hotel
            )
        )

        authRequest.approvedAt = Instant.now()
        invitationRepository.saveAndFlush(authRequest)
        return TApproveAuthRsp.newBuilder().setStatus(EApproveAuthStatus.AAS_OK).build()
    }

    private fun createUser(credentials: UserCredentials): User {
        val user = User(credentials.parsedPassportIdOrNull, credentials.login)
        return userRepository.saveAndFlush(user)
    }

    private fun createHotelWithOrg(hotelIdentifier: HotelIdentifier): Hotel {
        val hotelInfo = hotelConnectionService.getByHotelIdKey(hotelIdentifier)
            ?: throw IllegalStateException("Hotel is not found when approving the invitation")
        val org = Organization("Отель ${hotelInfo.getTitle()}")
        val hotel = Hotel(
            org,
            hotelInfo.hotelName,
            HotelManagementSource.fromPartnerId(hotelIdentifier.partnerId),
            hotelIdentifier,
        )
        hotel.permalink = hotel.permalink
        org.hotels.add(hotel)
        return organizationRepository.saveAndFlush(org).hotels.first()
    }

    private fun acceptInvitationPrecondition(userCredentials: UserCredentials, request: TApproveAuthReq) {

        try {
            Preconditions.checkArgument(
                !Strings.isNullOrEmpty(request.token),
                "request.token must not be null or empty"
            )
            Preconditions.checkArgument(
                userCredentials.parsedPassportIdOrNull != null,
                "userCredentials.passportId must be a valid numeric object"
            )
            Preconditions.checkArgument(
                !Strings.isNullOrEmpty(userCredentials.login),
                "userCredentials.login must not be null or empty"
            )
        } catch (e: Throwable) {
            throw Status.INVALID_ARGUMENT.withCause(e).withDescription(e.message).asException()
        }
    }
}

data class AuthRequestEmailData(
    val username: String,
    val token: String,
    val title: String,
)
