package ru.yandex.travel.hotels.extranet.service.notification

import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.PropertyNamingStrategy
import org.asynchttpclient.Realm
import org.asynchttpclient.RequestBuilder
import org.asynchttpclient.Response
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Service
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper
import java.io.IOException
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutionException
import java.util.concurrent.TimeoutException

@Service
@EnableConfigurationProperties(MailSenderConfigurationProperties::class)
class MailSenderServiceImpl(
    private val config: MailSenderConfigurationProperties,
    @Qualifier("mailerAsyncHttpClientWrapper") private val ahcClient: AsyncHttpClientWrapper,
) : MailSenderService {
    private val log = LoggerFactory.getLogger(javaClass)
    private val mapper: ObjectMapper = ObjectMapper()
        .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
        .setSerializationInclusion(JsonInclude.Include.NON_NULL)
        .setDefaultPropertyInclusion(JsonInclude.Include.NON_NULL)

    override fun sendEmailSync(
        targetEmail: String,
        campaign: String,
        data: JsonNode,
        attachments: List<MailSenderAttachment>,
    ) {
        sync(sendEmailAsync(targetEmail, campaign, data, attachments))
    }

    private fun sendEmailAsync(
        targetEmail: String,
        campaign: String,
        args: JsonNode,
        attachments: List<MailSenderAttachment>,
        async: Boolean = true, // асинхронная отправка на стороне рассылятора (ответ от ручки придет сразу, не ожидая отправки)
        countdown: Int? = null, // для асинхронной отправки - отправить рассылку не сразу, а через countdown секунд.
        expires: Int? = null, // для асинхронной отправки - не отправлять через expires секунд. через какое время уже не нужно отправлять письмо
    ): CompletableFuture<Response> {
        if (config.debug.enabled && config.debug.failEmail == targetEmail) {
            return CompletableFuture.failedFuture(RuntimeException("Exception for testing purposes"))
        }
        val realm = Realm.Builder(config.authenticationKey, "")
            .setScheme(Realm.AuthScheme.BASIC)
            .setUsePreemptiveAuth(true)
            .build()
        // old configs used to set campaign ids as "<CAMPAIGN_ID>/send", now we support proper ids without this suffix
        val actionSuffix = if (campaign.endsWith("/send")) "" else "/send"
        val url: String = config.mailerUrlBase + campaign + actionSuffix
        val requestBuilder = RequestBuilder()
            .setMethod("POST")
            .setUrl(url)
            .setRealm(realm)
            .addQueryParam("to_email", targetEmail)
            .addFormParam("args", mapper.writeValueAsString(args))
            .addFormParam("async", if (async) "true" else "false")
        if (attachments.isNotEmpty()) {
            requestBuilder.addFormParam("attachments", mapper.writeValueAsString(attachments))
        }
        if (countdown != null) {
            requestBuilder.addFormParam("countdown", countdown.toString())
        }
        if (expires != null) {
            requestBuilder.addFormParam("expires", expires.toString())
        }
        return ahcClient.executeRequest(requestBuilder)
    }

    private fun sync(future: CompletableFuture<Response>): Response {
        return try {
            val response = future.get()
            val statusCode = response!!.statusCode
            if (statusCode in 200..299) {
                response
            } else if (statusCode >= 500) {
                throw MailSenderRetryableException(
                    String.format(
                        "Exception happened while executing http call: %s. Response body: %s",
                        response.statusText, response.responseBody
                    )
                )
            } else {
                throw RuntimeException(
                    String.format(
                        "Exception happened while executing http call: %s. Response body: %s",
                        response.statusText, response.responseBody
                    )
                )
            }
        } catch (e: InterruptedException) {
            this.log.error("Mailer call interrupted", e)
            Thread.currentThread().interrupt() // preserved interruption status
            throw MailSenderRetryableException(e)
        } catch (e: ExecutionException) {
            if (e.cause == null) {
                throw RuntimeException("No root cause for ExecutionException found", e)
            }
            when (val cause = e.cause) {
                is TimeoutException -> {
                    throw MailSenderRetryableException(e)
                }
                is IOException -> {
                    throw MailSenderRetryableException(e)
                }
                is RuntimeException -> {
                    throw cause
                }
                else -> {
                    throw RuntimeException(e)
                }
            }
        }
    }
}
