package ru.yandex.travel.hotels.extranet.service.orders

import io.grpc.StatusException
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.Sort
import org.springframework.data.jpa.domain.Specification
import org.springframework.stereotype.Service
import ru.yandex.travel.commons.proto.ProtoUtils
import ru.yandex.travel.hotels.extranet.TBankOrderInfo
import ru.yandex.travel.hotels.extranet.TBoYOrderInfo
import ru.yandex.travel.hotels.extranet.TGetOrdersReq
import ru.yandex.travel.hotels.extranet.TGetOrdersRsp
import ru.yandex.travel.hotels.extranet.TGuest
import ru.yandex.travel.hotels.extranet.TPriceBreakdown
import ru.yandex.travel.hotels.extranet.entities.HotelIdentifier
import ru.yandex.travel.hotels.extranet.entities.orders.BoYOrderWebView
import ru.yandex.travel.hotels.extranet.repository.OrdersWebViewRepository
import ru.yandex.travel.hotels.extranet.service.betweenPredicateDate
import ru.yandex.travel.hotels.extranet.service.betweenPredicateDateTime
import ru.yandex.travel.hotels.extranet.service.hotels.HotelInfoService
import ru.yandex.travel.hotels.extranet.service.toPageable
import ru.yandex.travel.hotels.extranet.service.toTPrice
import java.math.BigDecimal
import javax.persistence.criteria.CriteriaBuilder
import javax.persistence.criteria.CriteriaQuery
import javax.persistence.criteria.Predicate
import javax.persistence.criteria.Root

@Service
class OrdersServiceImpl @Autowired constructor(
    private val repo: OrdersWebViewRepository,
    private val hotelInfoService: HotelInfoService,
) : OrdersService {
    @Throws(StatusException::class)
    override fun getOrders(req: TGetOrdersReq): TGetOrdersRsp {
        val rsp = TGetOrdersRsp.newBuilder()
        rsp.hotelInfo = hotelInfoService.getHotelInfo(HotelIdentifier.fromProto(req.hotelId))
        return repo.findAll(
            GetSpecImpl(req),
            req.page.toPageable(req.sortList, Sort.by(Sort.Direction.DESC, "checkInDate"))
        ).let { queryResult ->
            rsp.totalRecords = queryResult.totalElements
            queryResult.forEach {
                rsp.addResult(mapOrderToProto(it))
            }
            rsp.fiscalSum = queryResult.sumOf { it.fiscalPrice }.toTPrice()
            rsp.feeSum = queryResult.sumOf { it.feeAmount ?: BigDecimal.ZERO }.toTPrice()
            rsp.partnerSum = queryResult.sumOf { it.partnerAmount ?: BigDecimal.ZERO }.toTPrice()
            rsp.paidSum = queryResult.sumOf { it.paidAmount ?: BigDecimal.ZERO }.toTPrice()
            rsp.build()
        }
    }

    private fun mapOrderToProto(order: BoYOrderWebView): TBoYOrderInfo {
        val proto = TBoYOrderInfo.newBuilder()
        proto.id = order.id
        proto.prettyId = order.prettyId
        order.partnerOrderId?.let { proto.partnerOrderId = it }
        order.guestFirstName?.let {
            val guest = TGuest.newBuilder()
            guest.firstName = order.guestFirstName
            guest.lastName = order.guestLastName!!
            guest.build()
        }?.let {
            proto.firstGuest = it
        }
        proto.status = order.orderStatus
        proto.statusLocalized = order.orderStatusRus

        proto.createdAt = ProtoUtils.fromInstant(order.createdAt)
        proto.updatedAt = ProtoUtils.fromInstant(order.updatedAt)
        order.cancelledAt?.let { proto.cancellationDateTime = ProtoUtils.fromInstant(it) }
        proto.checkInDate = ProtoUtils.toTDate(order.checkInDate)
        proto.checkOutDate = ProtoUtils.toTDate(order.checkOutDate)
        proto.priceBreakdown = mapPriceBreakdown(order)

        order.bankOrdersArray?.forEach {
            proto.addBankOrderInfo(
                TBankOrderInfo.newBuilder()
                    .setBankOrderId(it)
                    .build()
            )
        }
        return proto.build()
    }

    private fun mapPriceBreakdown(order: BoYOrderWebView): TPriceBreakdown {
        val builder = TPriceBreakdown.newBuilder()
        builder.fiscalPrice = order.fiscalPrice.toTPrice()
        builder.hotelPrice = order.hotelPrice.toTPrice()
        builder.discount = order.discount.toTPrice()
        builder.fee = order.feeAmount.toTPrice()
        builder.partner = order.partnerAmount.toTPrice()
        builder.paidAmount = order.costAfterReservation.toTPrice()
        order.payoutAt?.let {
            builder.payoutAt = ProtoUtils.fromInstant(it)
        }
        return builder.build()
    }

    private class GetSpecImpl(val proto: TGetOrdersReq) : Specification<BoYOrderWebView> {
        /**
         * The approximate query being generated (with no optional filters and pagination)
         * <pre>
         * SELECT * FROM BoYOrder o
         *   JOIN HotelAgreement a on o.hotel_agreement_id = ? AND a.partner_id = ?
         * </pre>
         * */
        override fun toPredicate(
            root: Root<BoYOrderWebView>,
            query: CriteriaQuery<*>,
            builder: CriteriaBuilder
        ): Predicate {
            val list = mutableListOf(
                builder.equal(root.get<HotelIdentifier>("hotelId"), HotelIdentifier.fromProto(proto.hotelId))
            )
            if (proto.hasCheckInDate()) {
                list.add(
                    betweenPredicateDate(builder, root.get("checkInDate"), proto.checkInDate)
                )
            }
            if (proto.hasCheckOutDate()) {
                list.add(
                    betweenPredicateDate(builder, root.get("checkOutDate"), proto.checkOutDate)
                )
            }
            if (proto.hasOrderedAt()) {
                list.add(
                    betweenPredicateDateTime(builder, root.get("createdAt"), proto.orderedAt)
                )
            }
            if (proto.hasCancelledAt()) {
                list.add(
                    betweenPredicateDateTime(builder, root.get("cancelledAt"), proto.cancelledAt)
                )
            }
            return builder.and(*list.toTypedArray())
        }
    }
}
