package ru.yandex.travel.hotels.extranet.service.roles

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.hotels.extranet.IDMRole
import ru.yandex.travel.hotels.extranet.IDMRoleType
import ru.yandex.travel.hotels.extranet.IDMRoleUser
import ru.yandex.travel.hotels.extranet.IdmResponse
import ru.yandex.travel.hotels.extranet.LocalizedName
import ru.yandex.travel.hotels.extranet.entities.Role
import ru.yandex.travel.hotels.extranet.entities.User
import ru.yandex.travel.hotels.extranet.entities.UserRoleBinding
import ru.yandex.travel.hotels.extranet.repository.UserRepository
import ru.yandex.travel.hotels.extranet.repository.UserRoleBindingsRepository
import ru.yandex.travel.hotels.extranet.service.blackbox.BlackBoxService
import javax.transaction.Transactional

@Service
class IdmServiceImpl @Autowired constructor(
    private val blackBoxService: BlackBoxService,
    private val userRepository: UserRepository,
    private val userRoleBindingRepository: UserRoleBindingsRepository,
) : IdmService {
    override fun roleInfo(): List<IDMRole> {
        return Role.values().filter { it.isIdmManaged }.map {
            IDMRole.newBuilder()
                .setType(it.idmRoleType)
                .setName(
                    LocalizedName.newBuilder()
                        .setNameRu(it.displayNameRu)
                        .setNameEn(it.displayNameEn)
                        .build()
                )
                .build()
        }
    }

    @Transactional
    override fun grantIdmRole(roleType: IDMRoleType, login: String, externalLogin: String): IdmResponse {
        val uuid = blackBoxService.getUidByLogin(externalLogin)
            ?: return IdmResponse.newBuilder().setCode(403).setFatal("Unable to get passport uid for idm-provided login").build()

        val user = userRepository.findById(uuid).orElseGet {
            userRepository.save(User(uuid, externalLogin))
        }
        if (user.roles.any { it.role == Role.fromIdmRoleType(roleType) && it.idmLogin == login }) {
            return IdmResponse.newBuilder().setCode(1).setWarning("Role already exists").build()
        }
        user.roles.add(UserRoleBinding(user, Role.fromIdmRoleType(roleType), idmLogin = login))
        return IdmResponse.newBuilder().setCode(0).build()
    }

    @Transactional
    override fun removeIdmRole(roleType: IDMRoleType, login: String): IdmResponse {
        val role = Role.fromIdmRoleType(roleType)
        var done = false
        userRoleBindingRepository.findAllByRoleAndIdmLogin(role, login)
            .forEach { urb -> urb.user.roles.removeIf { it.role == role }.also { removed -> if (removed) done = true } }
        val builder = IdmResponse.newBuilder().setCode(0)
        if (!done) {
            builder.warning = "Role is already revoked"
        }
        return builder.build()
    }

    @Transactional
    override fun listIdmRoles(): List<IDMRoleUser> {
        return userRoleBindingRepository.findAllByIdmLoginIsNotNull().filter { it.role.isIdmManaged }.map {
            IDMRoleUser.newBuilder()
                .setType(it.role.idmRoleType)
                .setLogin(it.idmLogin)
                .setPassportLogin(it.user.login)
                .build()
        }
    }
}
