package ru.yandex.travel.hotels.extranet.service.roles

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.credentials.UserCredentials
import ru.yandex.travel.hotels.extranet.dto.OrganizationUserRoleDTO
import ru.yandex.travel.hotels.extranet.entities.HotelIdentifier
import ru.yandex.travel.hotels.extranet.entities.Permission
import ru.yandex.travel.hotels.extranet.entities.Role
import ru.yandex.travel.hotels.extranet.entities.User
import ru.yandex.travel.hotels.extranet.entities.UserRoleBinding
import ru.yandex.travel.hotels.extranet.errors.AuthorizationException
import ru.yandex.travel.hotels.extranet.errors.IllegalOperationException
import ru.yandex.travel.hotels.extranet.repository.HotelRepository
import ru.yandex.travel.hotels.extranet.repository.OrganizationRepository
import ru.yandex.travel.hotels.extranet.repository.UserRepository
import ru.yandex.travel.hotels.extranet.repository.UserRoleBindingsRepository
import java.util.UUID
import javax.transaction.Transactional

@Service
class UserRoleServiceImpl @Autowired constructor(
    private val organizationRepository: OrganizationRepository,
    private val userRepository: UserRepository,
    private val hotelRepository: HotelRepository,
    private val userRoleBindingsRepository: UserRoleBindingsRepository,
) : UserRoleService {

    @Transactional
    override fun grantRole(userId: Long, organizationId: UUID, role: Role, hotelId: Long?): UUID {
        checkPermission(Permission.MANAGE_USERS, organizationId)

        if (!role.allowsHotelLevel && hotelId != null) {
            throw IllegalOperationException("Role does not support hotel-level bindings")
        }
        if (role.isIdmManaged) {
            throw IllegalOperationException("Role does not support user-assignment")
        }

        val org = organizationRepository.getOne(organizationId)

        val user = userRepository.getOne(userId)
        if (user.blocked) {
            throw AuthorizationException("User is blocked")
        }

        val hotel = hotelId?.let {
            hotelRepository.getOne(it)
        }
        if (user.roles.any { it.role == role && it.organization == org && (it.hotel == null || it.hotel == hotel) }) {
            throw IllegalOperationException("Role already exists")
        }

        val binding = UserRoleBinding(user, role, organization = org, hotel = hotel)
        user.roles.add(binding)
        return binding.id
    }

    @Transactional
    override fun listOrgUsers(organizationId: UUID): List<OrganizationUserRoleDTO> {
        checkPermission(Permission.VIEW_USERS, organizationId)
        return organizationRepository.getOne(organizationId)
            .users
            .filter { !it.role.isIdmManaged && !it.user.blocked }
            .map { binding ->
                val builder = OrganizationUserRoleDTO.newBuilder()
                    .setId(binding.id.toString())
                    .setLogin(binding.user.login)
                    .setRole(binding.role.displayNameRu)
                binding.hotel?.name?.let { builder.setHotelName(it) }
                builder.build()
            }
    }

    @Transactional
    override fun revokeRole(id: UUID) {
        val binding = userRoleBindingsRepository.getOne(id)
        if (binding.organization == null) {
            throw IllegalOperationException("Unable to revoke global role")
        }
        if (binding.role.isIdmManaged) {
            throw IllegalOperationException("Unable to revoke idm role")
        }
        checkPermission(Permission.MANAGE_USERS, organizationId = binding.organization?.id).also {
            if (it == binding.user) {
                throw IllegalOperationException("Unable to revoke user's own role")
            }
        }
        userRoleBindingsRepository.delete(binding)
    }

    @Transactional(Transactional.TxType.REQUIRED)
    override fun checkPermission(
        permission: Permission, organizationId: UUID?,
        hotelId: Long?,
        hotelPartnerId: HotelIdentifier?
    ): User {
        val user = getCurrentUserOrThrow()
        var orgId = organizationId
        var hId = hotelId
        if (user.blocked) {
            throw AuthorizationException("user is blocked")
        }
        if (organizationId == null && hotelId == null && hotelPartnerId == null) {
            throw IllegalArgumentException("At least one org or hotel id should be specified")
        }
        if (hotelPartnerId != null) {
            val hotel = hotelRepository.findHotelByPartnerHotelId(hotelPartnerId)
                ?: throw AuthorizationException("hotel not found")
            if (hId != null && hId != hotel.id) {
                throw AuthorizationException("Incorrect combination of hotel ids")
            }
            hId = hotel.id
            if (orgId != null && orgId != hotel.organization.id) {
                throw AuthorizationException("Incorrect combination of hotel and org id")
            }
            orgId = hotel.organization.id
        } else if (hId != null && orgId == null) {
            orgId =
                hotelRepository.findById(hId).orElseThrow { AuthorizationException("hotel not found") }.organization.id
        }

        if (user.roles.none {
                it.role.permissions.contains(permission) &&
                    (it.role.isIdmManaged || it.organization?.id == orgId) &&
                    (it.role.isIdmManaged || it.hotel == null || it.hotel?.id == hId)
            }) {
            throw AuthorizationException("not authorized for $permission in org=$orgId for hotel $hId")
        }
        return user
    }

    override fun getCurrentUser(): User? {
        return UserCredentials.get()?.parsedPassportIdOrNull?.let {
            userRepository.findById(it).orElse(null)
        }
    }

    override fun getCurrentUserOrThrow(): User {
        return getCurrentUser() ?: throw AuthorizationException("unauthenticated")
    }
}
