package ru.yandex.travel.hotels.extranet.validation

import ru.yandex.travel.hotels.extranet.entities.AnnulationPenaltyType
import ru.yandex.travel.hotels.extranet.entities.AnnulationRule
import ru.yandex.travel.hotels.extranet.utils.isInteger
import java.math.BigDecimal
import javax.validation.ConstraintValidator
import javax.validation.ConstraintValidatorContext

class ValidAnnulationRuleItemValidator : ConstraintValidator<ValidAnnulationRuleItem, AnnulationRule> {
    override fun isValid(rule: AnnulationRule, context: ConstraintValidatorContext): Boolean {
        val penaltyNominal = rule.penaltyNominal
        when (rule.penaltyType) {
            AnnulationPenaltyType.FULL -> {
                if (penaltyNominal != null) {
                    return customError(
                        context,
                        "Нельзя указывать размер штрафа для правил, при которых аннуляция невозможна"
                    )
                }
            }
            AnnulationPenaltyType.NONE -> {
                if (penaltyNominal != null) {
                    return customError(context, "Нельзя указывать размер штрафа для правил без штрафа")
                }
            }
            AnnulationPenaltyType.NIGHTS -> {
                if (penaltyNominal == null) {
                    return customError(context, "Не указан размер штрафа")
                }
                if (!penaltyNominal.isInteger()) {
                    return customError(context, "Нецелое количество ночей")
                }
                if (penaltyNominal.signum() < 0) {
                    return customError(context, "Некорректное количество ночей")
                }
            }
            AnnulationPenaltyType.PERCENTAGE -> {
                if (penaltyNominal == null) {
                    return customError(context, "Не указан размер штрафа")
                }
                if (penaltyNominal < BigDecimal.valueOf(0) || penaltyNominal > BigDecimal.valueOf(100)) {
                    return customError(context, "Некорректный размер штрафа")
                }
            }
            AnnulationPenaltyType.FIXED -> {
                if (penaltyNominal == null) {
                    return customError(context, "Не указан размер штрафа")
                }
            }
        }

        return true
    }

    private fun customError(context: ConstraintValidatorContext, message: String): Boolean {
        context.disableDefaultConstraintViolation()
        context.buildConstraintViolationWithTemplate(message).addConstraintViolation()
        return false
    }
}
