package ru.yandex.travel.hotels.extranet.validation

import ru.yandex.travel.hotels.extranet.entities.AnnulationPenaltyType
import ru.yandex.travel.hotels.extranet.entities.AnnulationRule
import java.math.BigDecimal
import javax.validation.ConstraintValidator
import javax.validation.ConstraintValidatorContext

class ValidAnnulationRulesValidator : ConstraintValidator<ValidAnnulationRules, MutableList<AnnulationRule>> {
    override fun isValid(rules: MutableList<AnnulationRule>?, context: ConstraintValidatorContext): Boolean {
        if (rules == null || rules.isEmpty()) {
            return customError(context, "Не указаны правила аннуляции")
        }
        if (rules
                .filter { rule -> rule.penaltyType != AnnulationPenaltyType.FULL && rule.penaltyType != AnnulationPenaltyType.NONE }
                .map { rule -> rule.penaltyType }
                .distinct()
                .count() > 1
        ) {
            return customError(context, "Разные типы штрафов в правилах одной политики")
        }
        val sortedRules = rules.sortedBy { rule -> rule.comparator() }
        sortedRules.zipWithNext { a, b ->
            if (a.comparator() == b.comparator()) {
                return customError(context, "Одинаковый момент начала у разных правил")
            }
        }

        if (!sortedRules
                .mapNotNull { rule ->
                    when (rule.penaltyType) {
                        AnnulationPenaltyType.FULL -> BigDecimal.valueOf(Long.MAX_VALUE)
                        AnnulationPenaltyType.NONE -> BigDecimal.ZERO
                        else -> rule.penaltyNominal
                    }
                }
                .zipWithNext { a, b -> a >= b }
                .all { it }
        ) {
            return customError(context, "Неравномерная последовательность штрафов")
        }
        return true
    }

    private fun customError(context: ConstraintValidatorContext, message: String): Boolean {
        context.disableDefaultConstraintViolation()
        context.buildConstraintViolationWithTemplate(message).addConstraintViolation()
        return false
    }
}
