# coding=utf-8
import base64
import contextlib
import datetime
import hashlib
import json
import logging
import math
import os.path
import sys
import traceback
import warnings
import time
from collections import defaultdict
from uuid import uuid4

import six
import travel.hotels.proto2.label_pb2 as label_pb2
from dateutil import parser as date_parser
from library.python import resource
from travel.hotels.feeders.lib import downloaders, parsers, helpers
from travel.hotels.feeders.lib.model import objects, enums, features_enums, available_features
from travel.hotels.lib.python import ytlib
from yt.transfer_manager.client import TransferManager
from yt.wrapper import YtClient
import yt.wrapper

from travel.hotels.feeders.lib.common.data_helpers import merge_photo_duplicates
from travel.hotels.feeders.lib.metrics import Metrics
from travel.hotels.feeders.lib.model.log_message_types import DataCheckWarning, DateFormatWarning, AmenityWarning, DebugWarning, StarsWarning
from travel.hotels.lib.python.versioned_process import VersionedProcess

LOG = logging.getLogger(__name__)


def save_raw(item_stream, table_path, separate_client=False):
    def wrap_stream():  # encapsulate yt schema of raw table
        index = 0
        for item, extra in item_stream:
            yield {
                "item": item,
                "info": extra,
                "original_feed_index": index,
                "uuid": uuid4().hex,
            }
            index += 1

    if separate_client:
        client = YtClient(proxy=yt.wrapper.config["proxy"]["url"], token=yt.wrapper.config["token"])
    else:
        client = yt.wrapper

    client.write_table(table_path, wrap_stream())


class StarParser(object):
    star_map = {
        0: features_enums.Star.unrated,
        1: features_enums.Star.one,
        2: features_enums.Star.two,
        3: features_enums.Star.three,
        4: features_enums.Star.four,
        5: features_enums.Star.five
    }

    @classmethod
    def parse_stars(cls, s):
        if s is None:
            return None
        if isinstance(s, six.string_types):
            try:
                s = math.ceil(float(s))
            except:
                pass
        if s in cls.star_map:
            return cls.star_map[s]
        else:
            warnings.warn(StarsWarning("Unknown \"stars\" field value: {}".format(helpers.format_to_text(s))))
            return None


class TimeEnumParser(object):
    @staticmethod
    def get_time_enum(t):
        if isinstance(t, str) and t.startswith('24'):
            return "24"
        dt = (date_parser.parse(t) - datetime.datetime.min).seconds
        rt = round(dt / 1800) * 30
        return "{:02d}".format(int(rt / 60)) + ("30" if rt % 60 else "")


class FeatureMapper(object):
    @staticmethod
    def get_features(features_keys, features_map):
        features = []
        unknown = []
        if type(features_keys) is six.binary_type:
            features_keys = [features_keys]
        for feature_key in features_keys:
            if feature_key in features_map:
                features += features_map[feature_key]
            else:
                unknown.append(feature_key)
        return features, unknown

    @staticmethod
    def map_features(features_map, facilities, rubric, debug=False):
        features, unknown = FeatureMapper.get_features(facilities, features_map)
        for feature, value in features:
            if feature.feature_name not in available_features.available_features[rubric]:
                warnings.warn(AmenityWarning("Rubric '{}' has no feature '{}'".format(
                    rubric.name, feature.feature_name)))
                continue
            if isinstance(value, six.string_types):
                value = helpers.to_unicode(value.strip())
                if feature.type_class in [int, bool]:  # fix integer values
                    if debug and ',' in value:
                        warnings.warn(DebugWarning("\",\" in integer value: {}".format(value)))
                    value = value.replace(',',
                                          '').strip()  # todo: check this logic. handles cases where expedia passes 'number of rooms' as 3,044
                if feature.type_class is bool:
                    if value.lower() == "false":
                        value = False
                    elif value.lower() == "true":
                        value = True
                    try:
                        value = float(value)  # handle bool("0") case
                    except ValueError:
                        value = bool(value)
                    except Exception as e:
                        message = "Unable to cast value '{}' into type '{}' for amenity '{}'\n Exception: '{}'".format(
                            helpers.format_to_text(value), feature.type_class.__name__, feature.feature_name, e)
                        warnings.warn(AmenityWarning(message))
                        continue
                if feature.type_class is not None:
                    try:
                        value = feature.type_class(value)
                    except Exception as e:
                        message = "Unable to cast value '{}' into type '{}' for amenity '{}'\n Exception: '{}'".format(
                            helpers.format_to_text(value), feature.type_class.__name__, feature.feature_name, e)
                        warnings.warn(AmenityWarning(message))
                        continue
            if feature.allow_multi:
                if feature.cast(value) not in [v['value'] for v in feature.values if v is not None]:  # deduplication
                    feature.add(value)
            else:
                if debug and feature.values is not None and feature.values['value'] != feature.cast(value):
                    warnings.warn(AmenityWarning(
                        "Conflicting value for feature {}: {} != {}".format(feature.feature_name,
                                                                            feature.values['value'],
                                                                            feature.cast(value))))
                feature.set(value)
        for itm in unknown:
            warnings.warn(AmenityWarning("Unknown amenity '{}'".format(helpers.format_to_text(itm))))


class DefaultLangsProvider(object):
    langs_map = {  # assume that first language is the main.
        "ru": ['ru', 'en'],
        'default': ['en']
    }

    room_language_priority = {
        'default': ['ru', 'en'],
    }

    def __init__(self):
        self._room_language_priority_builded = {country: {lang: langs.index(lang) for lang in langs} for country, langs in six.iteritems(self.room_language_priority)}

    def get_langs(self, country):
        return self.langs_map.get(country.lower(), self.langs_map['default'])

    def choose_room_language(self, country, lang1, lang2):
        self._require_string('country', country)
        self._require_string('lang1', lang1)
        self._require_string('lang2', lang2)

        country_lowered = country.lower()
        lang1_lowered = lang1.lower()
        lang2_lowered = lang2.lower()

        index = self._room_language_priority_builded.get(country_lowered, self._room_language_priority_builded['default'])
        if lang1_lowered not in index and lang2_lowered not in index:
            lang = self._choose_lang_lexicographically(lang1_lowered, lang2_lowered)
        else:
            lang = self._choose_lang_with_max_priority(index, lang1_lowered, lang2_lowered)

        if lang == lang1_lowered:
            return lang1
        else:
            return lang2

    def _choose_lang_lexicographically(self, lang1, lang2):
        if lang1 < lang2:
            return lang1
        else:
            return lang2

    def _choose_lang_with_max_priority(self, index, lang1, lang2):
        if index.get(lang1, len(index)) < index.get(lang2, len(index)):
            return lang1
        else:
            return lang2

    def _require_string(self, param_name, value):
        if not isinstance(value, six.string_types):
            raise ValueError('Illegal argument "{}", expected "{}" type but was "{}"'.format(param_name, six.string_types[0].__name__, type(value).__name__))


class Partner(VersionedProcess):
    name = 'partner'
    category_name = 'feeds'

    min_records_count_accepted = 0
    max_records_added_or_removed_count = None
    warn_records_count_change = None

    yt_run_parameters = dict()

    publish_features = True
    allowed_fields = ['_city', '_partner']
    hidden_fields = ['addressAdd', 'chainId', 'publishingStatus', 'workingTime']  # todo: 'actualizationTime' - hide or fill. publishingStatus?
    list_to_string_fields = ['email', '_chain']
    merge_localized_fields = ['name', 'address', '_city', '_description', 'roomTypes']
    merge_key_value_fields = ['feature', '_feature', ]
    merge_dict_fields = []

    country_by_code = json.loads(resource.find('/common_country_mapping.json'))
    country_code_by_name = {v: k for k, u in country_by_code.items() for v in u}

    langs_provider = DefaultLangsProvider()
    languages = list(set().union(*six.itervalues(langs_provider.langs_map)))
    merge_translations = False

    def get_langs(self, country):
        return self.langs_provider.get_langs(country)

    table_by_country = {
        'RU': 'russia',
        # 'IL': 'israel',
        # 'CI': 'cote_divoir',
        # 'FI': 'finland',
        # 'LV': 'latvia',
        # 'LT': 'lithuania',
        # 'CZ': 'czech_republic',
        # 'RO': 'romania',
        # 'SC': 'serbia',
        # 'AZ': 'azerbaijan',
        # 'AM': 'armenia',
        # 'GE': 'georgia',
        # 'UZ': 'uzbekistan',
        # 'KG': 'kyrgyzstan',
        # 'MD': 'moldova',
        # 'BY': 'belarus',
        # 'KZ': 'kazakhstan',
        # 'TR': 'turkey',
        # 'UA': 'ukraine',
    }

    has_apartments = True

    hotels_table_name = "hotels"
    distinguished_countries = list(table_by_country.keys())
    output_log_table_name = "_log"
    output_labels_table_name = "labels"
    time_format = "%H:%M"

    @staticmethod
    def parse_stars(s):
        return StarParser.parse_stars(s)

    def __init__(self, session, args):
        super(Partner, self).__init__(session, args)
        self.args = args
        self.limit = args.limit
        self.debug |= self.limit != parsers.NO_LIMIT  # if limit is set - activate debug mode
        self.skip_checks = args.skip_checks
        self.check_counts = args.check_counts
        self.teamcity_output = args.enable_teamcity_diagnostics
        self.metrics = Metrics(self.name, session.timelabel, self.get_run_dir(), args)
        self.distinguished_countries = args.distinguished_countries
        self.table_by_country = {c.upper(): self.table_by_country[c.upper()] for c in self.distinguished_countries}
        countries = list(six.itervalues(self.table_by_country)) + ["world", ]
        self.output_hotel_table_names = list()

        if self.has_apartments:
            self.table_by_rubric = {
                enums.HotelRubric.ACCOMMODATION_FOR_DAILY_RENT: "apartments",
            }  # all others -> 'hotels'
        else:
            self.table_by_rubric = {}  # all to "hotels"

        for country in countries:
            for rubric in list(six.itervalues(self.table_by_rubric)) + ['hotels', ]:
                self.output_hotel_table_names.append('_'.join([country, rubric]))
        extra_table_names = [self.output_labels_table_name, self.output_log_table_name, ]
        self.output_table_names = self.output_hotel_table_names + extra_table_names
        self.output_table_switches = {t: i for i, t in enumerate(self.output_table_names)}

        self.field_merge_functions = {
            'roomTypes': self.merge_room_types,
            'photos': self.merge_photos,
        }

    def get_raw_dir(self):
        return ytlib.join(self.get_run_dir(), "raw")

    def get_parsed_dir(self):
        return ytlib.join(self.get_run_dir(), "parsed")  # Only feeds here

    def get_parsed_aux_dir(self):
        return ytlib.join(self.get_run_dir(), "parsed_aux")  # Labels, logs...

    def get_raw_table_path(self, name):
        return ytlib.join(self.get_raw_dir(), name)

    def get_parsed_table_path(self, name):
        return ytlib.join(self.get_parsed_dir(), name)

    def get_parsed_aux_table_path(self, name):
        return ytlib.join(self.get_parsed_aux_dir(), name)

    @property
    def output_hotel_table_paths(self):
        return map(self.get_parsed_table_path, self.output_hotel_table_names)

    @property
    def output_labels_table_path(self):
        return self.get_parsed_aux_table_path(self.output_labels_table_name)

    @property
    def output_log_table_path(self):
        return self.get_parsed_aux_table_path(self.output_log_table_name)

    def get_output_table_paths(self):
        parsed_dir = self.get_parsed_dir()
        return [ytlib.join(parsed_dir, table) for table in self.output_table_names]

    def prepare(self):
        ytlib.ensure_dir_exists(self.get_raw_dir())
        ytlib.ensure_dir_exists(self.get_parsed_dir())
        ytlib.ensure_dir_exists(self.get_parsed_aux_dir())
        for table_path in self.output_hotel_table_paths:
            ytlib.yt.create("table", table_path, attributes={"schema": objects.Hotel.get_yt_schema(partner_name=self.name,
                                                                                                   publish_features=self.publish_features,
                                                                                                   hidden_fields=self.hidden_fields,
                                                                                                   allowed_fields=self.allowed_fields)})

        ytlib.yt.create("table", self.output_labels_table_path, attributes={"schema": [
            {"name": "Label", "type": "string"},
            {"name": "Proto", "type": "string"},
        ]})

        ytlib.yt.create("table", self.output_log_table_path, attributes={"schema": [
            {"name": "timestamp", "type": "int64"},
            {"name": "level", "type": "string"},
            {"name": "type", "type": "string"},
            {"name": "message", "type": "string"},
            {"name": "original_id", "type": "string"},
            {"name": "uuid", "type": "string"},
            {"name": "country", "type": "string"},
            {"name": "rubric", "type": "string"},
        ]})

    def map(self, item, info):
        raise NotImplementedError

    def map_raw_table(self, table_name, show_warnings=True):
        def warn_mapper(row):
            with warnings.catch_warnings(record=True) as ws:
                warnings.simplefilter("always")
                record = self.map(row["item"], row["info"])
                if record is not None:
                    try:
                        objects.check_data(record, debug=self.debug)
                    except Exception as e:
                        warnings.warn(DataCheckWarning("'check_data' failed with exception: '{}'".format(e)))
                if not ws:
                    ws = []
                return ws, record

        @ytlib.yt.with_context
        def yt_mapper(row, context):
            original_id = None
            country = None
            rubric = None

            uuid = row.get("uuid")
            timestamp = int(time.time())
            try:
                ws, record = warn_mapper(row)
                if show_warnings and len(ws) > 0:
                    yield ytlib.yt.create_table_switch(self.output_table_switches['_log'])
                    original_id = None if record is None else str(record.original_id)
                    country = None if record is None else str(record.country)
                    rubric = None if record is None else str(helpers.extract_value_from_list(record.rubric.values)['value'])
                    for w in ws:
                        yield {
                            "timestamp": timestamp,
                            "level": "warning",
                            "type": w.category.__name__,
                            "message": str(w.message),
                            "original_id": original_id,
                            "uuid": uuid,
                            "country": country,
                            "rubric": rubric,
                        }
            except Exception as e:
                message = traceback.format_exc()
                yield ytlib.yt.create_table_switch(self.output_table_switches['_log'])
                yield {
                    "timestamp": timestamp,
                    "level": "error",
                    "type": type(e).__name__,
                    "message": message,
                    "original_id": original_id,
                    "uuid": uuid,
                    "country": country,
                    "rubric": rubric,
                }
                record = None
            if record is None:
                return
            yield ytlib.yt.create_table_switch(self.output_table_switches[self.output_labels_table_name])
            label_row = {
                "Label": record.label_hash,
                "Proto": record.label_proto,
            }
            yield label_row
            if self.output_hotel_table_paths:
                country = self.table_by_country.get(helpers.extract_value_from_list(record.country.values), 'world')
                rubric = self.table_by_rubric.get(
                    enums.HotelRubric(helpers.extract_value_from_list(record.rubric.values)['value']), 'hotels'
                )
                output_table_name = '_'.join([country, rubric])
                yield ytlib.yt.create_table_switch(self.output_table_switches[output_table_name])
            yield record.to_dict(
                partner_name=self.name,
                publish_features=self.publish_features,
                hidden_fields=self.hidden_fields,
                allowed_fields=self.allowed_fields,
                list_to_string_fields=self.list_to_string_fields
            )

        input_table = self.get_raw_table_path(table_name)
        extra_table_paths = [self.output_labels_table_path, self.output_log_table_path]
        output_tables = self.output_hotel_table_paths + extra_table_paths
        with ytlib.hide_sys_args():
            spec = {'auto_merge': {'mode': 'relaxed'}, "job_io": {"control_attributes": {"enable_row_index": True}}}
            ytlib.yt.run_map(yt_mapper, input_table, output_tables, spec=spec, **self.yt_run_parameters)
            for table in self.output_hotel_table_paths:
                ytlib.yt.run_sort(table, sort_by="originalId")
                if self.merge_translations:
                    # todo: reduce only countries we have translations in.
                    ytlib.yt.run_reduce(
                        self.get_localization_merge_reduce(),
                        source_table=table,
                        destination_table=table,
                        reduce_by=["originalId"],
                        sort_by="originalId",
                        **self.yt_run_parameters)

    def download_all_feeds(self):
        pass

    def process_all_feeds(self):
        self.map_raw_table("hotels")

    def check_results(self):
        if self.skip_checks:
            LOG.info("Skipping checks. Results were published at %s", ytlib.get_url(self.get_run_dir()))
            return
        status_bits = ""
        html_bits = """
        <!DOCTYPE html><html lang="en">
        <head><link rel="stylesheet" href="//yastatic.net/bootstrap/3.3.6/css/bootstrap.min.css"/></head>
        <body><div class="container">
        """
        # Here i assume that this check actually always happens - because previous feed exists.
        # so we always will check total quantity of records.
        total_records = 0
        added_records = 0
        removed_records = 0

        event_counts = ytlib.count_events(self.output_log_table_path)
        for tag, value in event_counts.items():
            self.metrics.send("event_counts", value, tag=tag)
        if event_counts["error"] > 0:
            status_bits += "Fail"
            html_bits += '<h3 class="text-danger">Fail</h3>'
        else:
            status_bits += "OK"
            html_bits += '<h3 class="text-success">OK</h3>'
        status_bits += "; {error}E/{warning}W".format(**event_counts)
        html_bits += "<p>{error} errors; {warning} warnings.</p>".format(**event_counts)
        html_bits += '<p><a href="{0}" target="_blank">Open current session in YT</a> ({1})</p>'.format(
            ytlib.get_url(self.get_run_dir()), self.get_run_dir())

        if ytlib.yt.exists(self.get_latest_path()):
            previous_base_dir = ytlib.yt.get(ytlib.join(self.get_latest_path(), '@path'))
            LOG.info("Comparing results with the previous session: %s", previous_base_dir)
            html_bits += """
            <p><a href="{0}" target="_blank">Open previous session in YT</a> ({1})</p>
            <table class="table">
            <thead><tr><th>&nbsp;</th><th>Total</th><th>Added</th><th>Removed</th><th>Changed</th></tr></thead>
            <tbody>
            """.format(ytlib.get_url(previous_base_dir), previous_base_dir)
            for table_name in self.output_hotel_table_names:
                LOG.info("Comparing results in '%s'", table_name)
                previous_table_path = ytlib.join(previous_base_dir, "parsed", table_name)
                if not ytlib.yt.exists(previous_table_path):
                    LOG.info("Previous table '%s' doesn't exist, skipping comparison.", previous_table_path)
                    continue
                current_table_path = self.get_parsed_table_path(table_name)
                diff, fields = ytlib.diff_tables(previous_table_path, current_table_path)
                for tag, value in diff.items():
                    self.metrics.send("hotel_counts", value, scope=table_name, tag=tag)
                for field, value in fields.items():
                    self.metrics.send("hotel_counts", value, scope=table_name, tag="changed", field=field)
                status_bits += "; {}".format(table_name)
                status_bits += " (+{added}, -{removed}, ^{changed}, ={total})".format(**diff)
                html_bits += "<tr>"
                html_bits += "<td>{}</td>".format(table_name)
                html_bits += "<td>{total}</td><td>{added}</td><td>{removed}</td><td>{changed}</td>".format(**diff)
                html_bits += "</tr>"

                added_records += diff['added']
                removed_records += diff['removed']

            html_bits += """
            </tbody></table>
            """

        html_bits += """
        </div></body>
        </html>
        """

        for table_name in self.output_hotel_table_names:
            current_table_path = self.get_parsed_table_path(table_name)
            total_records += ytlib.yt.get_attribute(current_table_path, "row_count")

        if self.teamcity_output:
            six.print_("##teamcity[buildStatus text='{}']".format(status_bits), file=sys.stderr)
            report_path = os.path.abspath("test_results.html")
            with open(report_path, "w") as handle:
                handle.write(html_bits)
            six.print_("##teamcity[publishArtifacts '{}']".format(report_path), file=sys.stderr)

        if event_counts["error"] > 0:
            message = "There were errors while mapping the feed; see {}".format(
                ytlib.get_url(self.output_log_table_path))
            if self.teamcity_output:
                six.print_("##teamcity[buildProblem description='{}']".format(message))
            raise Exception(message)

        if self.check_counts and self.limit == parsers.NO_LIMIT:
            if total_records < self.args.min_records_count_accepted:
                message = "Too few records in feed of partner {}: {} < {}; see {}".format(
                    self.name, total_records, self.args.min_records_count_accepted, ytlib.get_url(self.get_run_dir()))
                if self.teamcity_output:
                    six.print_("##teamcity[buildProblem description='{}']".format(message))
                raise Exception(message)

            if self.args.max_records_added_or_removed_count is None:
                message = "{} records removed without check for {}".format(removed_records, self.name)
                warnings.warn(DataCheckWarning(message))
            elif max(removed_records, added_records) > self.args.max_records_added_or_removed_count:
                message = "Too many records added ({2}) or removed ({1}) in feed of partner {0}: > {3}; see {4}".format(
                    self.name, removed_records, added_records, self.args.max_records_added_or_removed_count, ytlib.get_url(self.get_run_dir()))
                if self.teamcity_output:
                    six.print_("##teamcity[buildProblem description='{}']".format(message))
                raise Exception(message)

            changed_records = removed_records + added_records
            if self.args.warn_records_count_change is not None and changed_records > self.args.warn_records_count_change:
                message = "A lot of records changed (added or removed) in feed of partner {}. {} > {}; see {}".format(
                    self.name, changed_records, self.args.warn_records_count_change, ytlib.get_url(self.get_run_dir()))
                warnings.warn(DataCheckWarning(message))
                if self.teamcity_output:
                    six.print_("##teamcity[buildStatus description='{}']".format(message))

    def create_feed_table(self, name, ignore_existing=False):
        ytlib.yt.create('table', self.get_raw_table_path(name), ignore_existing=ignore_existing)

    @contextlib.contextmanager
    def download_feed(self, url, downloader=None, parser=None, headers=None, auth=None, limit=None):
        if downloader is None:
            downloader = downloaders.StreamingDownloader()
        if parser is None:
            parser = parsers.XmlParser()
        with downloader.get(url, headers=headers, auth=auth) as feed:
            yield parser.parse(feed, limit=limit or self.limit)

    def download_and_save(self, url, name, downloader=None, parser=None, headers=None, auth=None, limit=None, append=False, separate_client=False):
        with self.download_feed(url, downloader, parser, headers, auth, limit) as feed:
            self.save_raw_feed(feed, name, append=append, separate_client=separate_client)

    def save_raw_feed(self, feed, name, append=False, separate_client=False):
        save_raw(feed, ytlib.yt.TablePath(self.get_raw_table_path(name), append=append), separate_client=separate_client)

    def transfer_labels(self):
        yt_proxy = self.args.yt_proxy.split('.')[0]
        transfer_to_yt_proxy = self.args.transfer_to_yt_proxy.split('.')[0]
        source_pattern = self.output_labels_table_path
        destination_pattern = ytlib.join(self.get_latest_path(), "parsed_aux", "labels")
        transfer_manager = TransferManager(token=self.args.yt_token)
        transfer_manager.add_tasks(
            source_cluster=yt_proxy,
            source_pattern=source_pattern,
            destination_cluster=transfer_to_yt_proxy,
            destination_pattern=destination_pattern,
        )

    def run(self):
        self.prepare()
        LOG.info("Running...")
        with self.metrics.batch_send():
            with self.metrics.timed("download_feeds"):
                self.download_all_feeds()
            with self.metrics.timed("parse_feeds"):
                self.process_all_feeds()
            with self.metrics.timed("check_results"):
                self.check_results()
        if self.args.transfer_labels:
            self.transfer_labels()
        LOG.info("Done!")

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        common_group = parser.add_argument_group(Partner.name + " - common")
        common_group.add_argument("--enable-teamcity-diagnostics", action="store_true", default=False,
                                  help="produce extra diagnostics for TeamCity")
        common_group.add_argument("--limit", metavar="N", type=int, default=parsers.NO_LIMIT,
                                  help="limit downloaders to produce first N items. Activates debug mode.")
        common_group.add_argument("--skip_checks", '--sc', action="store_true", default=False,
                                  help="skip time-consuming checks for testing basic parsing")
        common_group.add_argument("--skip_counts_checks", '--scc', dest='check_counts', action="store_false", default=True,
                                  help="Skip records counts checks in feeds")
        common_group.add_argument("--distinguished_countries", '--dc', nargs='+', default=proc_env.caller_cls.distinguished_countries,
                                  help="Countries for separate output tables (by default everything goes to work")

        solomon_group = parser.add_argument_group(Partner.name + " - solomon integration")
        solomon_group.add_argument("--send-metrics-to-solomon", action="store_true", default=False)
        solomon_group.add_argument("--solomon-project", default="travel")
        solomon_group.add_argument("--solomon-service", default="feeders")
        solomon_group.add_argument("--solomon-cluster", default="dev-" + proc_env.user)
        solomon_group.add_argument("--solomon-token", default=None)

        statface_group = parser.add_argument_group(Partner.name + " - statface integration")
        statface_group.add_argument("--send-metrics-to-statface", action="store_true", default=False)
        statface_group.add_argument("--statface-user")
        statface_group.add_argument("--statface-password")

        labels_group = parser.add_argument_group(Partner.name + " - hotel labels")
        labels_group.add_argument("--transfer-labels", action="store_true", default=False)
        labels_group.add_argument("--transfer-to-yt-proxy", default="arnold")

        record_limits_group = parser.add_argument_group(Partner.name + " - record limits")
        record_limits_group.add_argument("--min-records-count-accepted", type=int,
                                         help='Minimum accepted record count',
                                         default=proc_env.caller_cls.min_records_count_accepted)
        record_limits_group.add_argument("--max-records-added-or-removed-count", type=int,
                                         help="Max record to be added or removed",
                                         default=proc_env.caller_cls.max_records_added_or_removed_count)
        record_limits_group.add_argument("--warn-records-count-change", type=int,
                                         help="Changed records count which causes warning",
                                         default=proc_env.caller_cls.warn_records_count_change)

    @staticmethod
    def configure_session(session, proc_env):
        session.annotate("partner", proc_env.caller_cls.name)
        session.annotate("limit", proc_env.args.limit)

    @staticmethod
    def get_label(partner_name, original_id):
        campaign = "ytravel_{}".format(partner_name)
        label_proto = label_pb2.TLabel(
            Source="sprav",
            Medium="feed_links",
            Campaign=campaign,
            OriginalHotelId=original_id
        )
        return label_proto.SerializeToString()

    @staticmethod
    def get_hash(label):
        h = hashlib.sha224(label).digest()
        return 's' + Partner.get_base64(h)

    @staticmethod
    def get_base64(value):
        return base64.urlsafe_b64encode(value).rstrip('=')

    @staticmethod
    def generate_label(hotel, partner_name):
        proto_decoded = Partner.get_label(partner_name, str(hotel.original_id))
        hotel.label_hash = Partner.get_hash(proto_decoded)
        hotel.label_proto = Partner.get_base64(proto_decoded)

    @classmethod
    def format_time(cls, t):
        if t is None:
            return None
        if t == '24:00':  # poor parser can't handle that
            t = '00:00'
        try:
            dt = date_parser.parse(t) if isinstance(t, six.string_types) else t
            return dt.strftime(cls.time_format)
        except:
            warnings.warn(DateFormatWarning("Unable to convert time {} into format {}".format(helpers.format_to_text(t), cls.time_format)))
            return None

    @staticmethod
    def get_time_enum(t):
        return TimeEnumParser.get_time_enum(t)

    def get_localization_merge_reduce(self):

        def lang_index(lang, langs):
            try:
                return langs.index(lang)
            except ValueError:
                return -1

        def value_converter(k, v):
            if isinstance(v, list):
                return k, tuple(v)
            else:
                return k, v

        def freeze(d):
            return frozenset(map(lambda i: value_converter(i[0], i[1]), six.iteritems(d)))

        def yt_reduce(key, input_row_iterator):
            # merging records in different languages
            result = dict(**key)  # saves 'originalId'
            for input_row in input_row_iterator:
                if result.get('actualizationTime') is not None and input_row.get('actualizationTime') is not None:
                    valid_update = result['actualizationTime'] < input_row['actualizationTime']
                else:
                    valid_update = None
                for field in six.iterkeys(input_row):
                    if field not in result or result[field] is None or result[field] == [] or result[field] == [None, ]:
                        result[field] = input_row[field]
                        continue
                    if input_row[field] is None or input_row[field] == [] or input_row[field] == [None, ]:
                        continue

                    if field in self.field_merge_functions:
                        result[field] = self.field_merge_functions[field](result, result[field], input_row[field])
                    elif field in self.merge_localized_fields:
                        if isinstance(input_row[field], list) and (input_row[field] != [None, ]):
                            result[field] += input_row[field]
                    elif field in self.merge_key_value_fields:
                        res = defaultdict(set)
                        inp = defaultdict(set)
                        for v in result[field]:
                            if v is None or (isinstance(v, dict) and 'id' not in v):
                                continue
                            res[v["id"]].add(freeze(v))
                        for v in input_row[field]:
                            if v is None or (isinstance(v, dict) and 'id' not in v):
                                continue
                            inp[v["id"]].add(freeze(v))
                        for v in input_row[field]:
                            if v is None or (isinstance(v, dict) and 'id' not in v):
                                continue
                            key = v["id"]
                            if freeze(v) not in res[key]:
                                if len(res[key]) != 1 or len(inp[key]) != 1:  # thus we will keep 1 item if there were 1 item and multiple if multiple
                                    result[field].append(v)  # adds any new values. Might cause issues with allow_multi=False
                                else:
                                    warnings.warn("Mismatch in non-localized data in field {}, key {}: {} != {}".format(field, key, list(res[key])[0], v))
                    elif field in self.merge_dict_fields:  # photos are slightly different for different localizations for some reason
                        res = {freeze(d) for d in result[field]}
                        for d in input_row[field]:
                            if freeze(d) not in res:
                                result[field].append(d)
                    elif result[field] != input_row[field]:  # unlocalized data matches
                        if valid_update is not None:
                            if valid_update:
                                result[field] = input_row[field]
                            continue
                        try:  # known case: lists can be arranged in different order
                            res = result[field]
                            inp = input_row[field]
                            assert isinstance(res, list) and len(res) == 1 and isinstance(res[0], list)
                            assert isinstance(inp, list) and len(inp) == 1 and isinstance(inp[0], list)
                            assert set(res[0]) == set(inp[0])
                            continue
                        except AssertionError:
                            pass
                        warnings.warn("Mismatch in non-localized data in field {}: {} != {}".format(field, result[field], input_row[field]))
                        if self.debug:
                            warnings.warn("Original record: {}".format(result))
                            warnings.warn("New record: {}".format(input_row))
            langs = self.get_langs(result['country'])
            for field in self.merge_localized_fields:
                if field in result and isinstance(result[field], list) and len(result[field]) > 1:
                    result[field] = sorted(result[field], key=lambda val: lang_index(val.get('lang', '').lower(), langs))
            yield result

        return yt_reduce

    def map_features(self, features_map, facilities, rubric, debug=False):
        FeatureMapper.map_features(features_map, facilities, rubric, debug)

    def merge_room_types(self, result, result_room_types, row_room_types):
        for room in row_room_types:
            room_type_with_same_id = next((x for x in result_room_types if x['id'] == room['id']), None)
            if room_type_with_same_id:
                country = result['country']
                current_room_lang = room_type_with_same_id['lang']
                new_room_lang = room['lang']

                lang = self.langs_provider.choose_room_language(country, current_room_lang, new_room_lang)
                if lang == new_room_lang:
                    index = result_room_types.index(room_type_with_same_id)
                    result_room_types[index] = room
                elif lang == current_room_lang:
                    pass
                else:
                    raise Exception('Incorrect language code selected {} must be {} or {}.'.format(lang, current_room_lang, new_room_lang))
            else:
                result_room_types.append(room)

        return result_room_types

    def merge_photos(self, result, result_photos, row_photos):
        return merge_photo_duplicates(result_photos + row_photos)
