# -*- coding: utf8 -*-
import cPickle as pickle
import os

import yt.wrapper as yt
from yt.wrapper.client import YtClient


def multireplace(s, replaces):
    for k, v in replaces:
        s = s.replace(k, v)
    return s


class AltayDb(object):
    snapshot_path = "//home/altay/db/export/current-state/snapshot"

    @staticmethod
    def format_feature_name(name):
        return '_'.join(multireplace(name.lower(), [('-', ' '), ('с', 'c')]).split())

    @staticmethod
    def to_camel_case(s):
        return ''.join(w.title() for w in s.replace('_', ' ').split())

    @staticmethod
    def format_class_name(name):
        return AltayDb.to_camel_case(name)

    @staticmethod
    def format_enum_name(name):
        replaces = [('-', '_'), ('сквош', 'squash'), ('.', '_'), ('с', 'c')]
        return multireplace('_'.join(name.lower().split()), replaces)

    @staticmethod
    def format_dict_key(s):
        return multireplace(s, [('"', '\\"'), ("'", "\\'")])

    @staticmethod
    def get_cached(getter, cache_file_path, clear_cache=False):
        def pload(path):
            with open(path, 'rb') as fh:
                return pickle.load(fh)

        def pdump(obj, path):
            with open(path, 'wb') as fh:
                return pickle.dump(obj, fh)

        if not clear_cache and os.path.exists(cache_file_path):
            return pload(cache_file_path)
        else:
            res = getter()
            dir_path = os.path.dirname(cache_file_path)
            if not os.path.exists(dir_path):
                os.makedirs(dir_path)
            pdump(res, cache_file_path)
            return res

    def __init__(self, use_cache=False, cache_path='./cache', clear_cache=False):
        yt_client = YtClient(proxy='hahn')

        def read_altay_dict(dict_name):
            return {item['id']: item for item in yt_client.read_table(yt.ypath_join(self.snapshot_path, dict_name))}

        def cached_read_altay_dict(dict_name):
            cache_file_path = os.path.join(cache_path, dict_name + '.pkl')
            return AltayDb.get_cached(lambda: read_altay_dict(dict_name), cache_file_path=cache_file_path, clear_cache=clear_cache)

        def get_altay_dict(dict_name):
            if use_cache:
                return cached_read_altay_dict(dict_name)
            else:
                return read_altay_dict(dict_name)

        self.rubrics = get_altay_dict('rubric')
        self.features = get_altay_dict('feature')
        self.enum_values = get_altay_dict('feature_enum_value')
        self.units = get_altay_dict('unit')
        self.unit_groups = get_altay_dict('unit_group')
        self.rubric_tree = {}
        self.extracted_features = {}
        self.feature_by_id = {}

    @staticmethod
    def validate_feature(feature):
        assert isinstance(feature["is_multiple"], bool)
        value_type = feature["value_type"]
        assert value_type is not None
        if value_type in ["logical_value", "number"]:
            assert feature["unit_group_id"] is None
            assert feature["time_interval_unit"] is None
            assert isinstance(feature["feature_group_ids"], list)
            assert len(feature["feature_group_ids"]) == 0
            assert not feature["is_multiple"]
        elif value_type == "dict_value":
            assert feature["unit_group_id"] is None
            assert feature["time_interval_unit"] is None
            assert isinstance(feature["feature_group_ids"], list)
            assert len(feature["feature_group_ids"]) == 0
        elif value_type == "feature_group":
            assert feature["unit_group_id"] is None
            assert feature["time_interval_unit"] is None
            assert isinstance(feature["feature_group_ids"], list)
            assert len(feature["feature_group_ids"]) > 0

    def extract_feature(self, feature):
        self.validate_feature(feature)
        value_type = feature["value_type"]
        if value_type == "dict_value":
            yield self.extract_dict_feature(feature)
        elif value_type == "feature_group":
            for extracted_feature in self.extract_feature_group_feature(feature):
                yield extracted_feature
        else:
            yield self.extract_simple_feature(feature)

    @staticmethod
    def extract_simple_feature(feature):
        extracted_feature = {}
        other_translations = []
        main_translation = None
        if 'names' in feature:
            for name in feature['names']:
                locale = None
                if isinstance(name['value'], dict):
                    value = name['value']['value']
                    if 'locale' in name['value']:
                        locale = name['value']['locale']
                else:
                    value = name['value']
                if name['type'] == 'main' and locale == 'ru':
                    main_translation = value
                else:
                    if locale in ['ru', 'en', None]:
                        other_translations.append(value)
        extracted_feature['main_translation'] = main_translation
        extracted_feature['other_translations'] = other_translations
        extracted_feature['id'] = feature['id']
        extracted_feature['is_multiple'] = feature['is_multiple']
        extracted_feature['value_type'] = feature['value_type']
        extracted_feature['permalink'] = feature['permalink']
        return extracted_feature

    def extract_dict_feature(self, feature):
        extracted_feature = self.extract_simple_feature(feature)
        values = {}
        value_translations = {}
        for enum_value_id in feature["enum_values"]:
            enum_value = self.enum_values[enum_value_id]
            if enum_value["publishing_status"] != "publish":
                continue
            values[enum_value['id']] = enum_value["permalink"]
            for name in enum_value['names']:
                translation = name['value']['value']
                if (translation in value_translations) and (value_translations[translation] != enum_value["permalink"]):
                    if feature['permalink'] not in ['hotel_price_category', ]:
                        print("conflicting translation in feature \"{}\" for \"{}\" and \"{}\": \"{}\"".format(
                            feature["permalink"],
                            value_translations[translation],
                            enum_value["permalink"],
                            translation
                        ))
                value_translations[translation] = enum_value["permalink"]
        extracted_feature['values'] = values
        extracted_feature['translations'] = value_translations
        return extracted_feature

    def extract_feature_group_feature(self, feature):
        for subfeature_id in feature["feature_group_ids"]:
            subfeature = self.features[subfeature_id]
            if subfeature["publishing_status"] != "publish":
                continue
            for extracted_feature in self.extract_feature(subfeature):
                yield extracted_feature

    def extract_rubric_features(self, rubric_id):
        assert isinstance(rubric_id, int)

        rubric_ids = set()
        while rubric_id is not None:
            rubric_ids.add(rubric_id)
            rubric_id = self.rubrics[rubric_id]["parent_rubric_id"]

        for rubric_id in rubric_ids:
            rubric = self.rubrics[rubric_id]
            if rubric["publishing_status"] != "publish":
                continue
            for rubric_feature in rubric["features"]:
                feature = self.features[rubric_feature["feature_id"]]
                if feature["publishing_status"] != "publish":
                    continue
                for extracted_feature in self.extract_feature(feature):
                    self.extracted_features[extracted_feature['permalink']] = extracted_feature
                    self.feature_by_id[extracted_feature['id']] = extracted_feature
                    yield extracted_feature

    def load_rubric_features(self, rubric_id):
        self.rubric_tree[rubric_id] = {feature['permalink']: feature for feature in self.extract_rubric_features(rubric_id)}
