# coding=utf-8

import json
import warnings
from collections import defaultdict

import library.python.resource as rs
import six
from travel.hotels.feeders.lib.model import enums, objects, features_enums

from travel.hotels.feeders.lib.helpers import wrap_into_list
from travel.hotels.feeders.lib.model.log_message_types import GeoWarning, AmenityWarning, DictionaryWarning

phone_map = {
    "fax": enums.PhoneType.FAX,
    "main": enums.PhoneType.PHONE,
    "phone": enums.PhoneType.PHONE,
}

russia_names = {"Россия", "Российская Федерация", "Russia", "Russian Federation"}


def get_lang(obj):
    try:
        return enums.Language[obj["@lang"].upper()]
    except KeyError:
        return None


country_mapping = json.loads(rs.find('/common_country_mapping.json'))
country_name_to_code = {val: k for k, v in six.iteritems(country_mapping) for val in wrap_into_list(v)}
country_name_to_code.update({v.encode('utf-8'): k for k, u in country_mapping.items() for v in u})


def map_backa(item, debug=False):
    hotel = objects.Hotel()
    names = wrap_into_list(item["name"])
    for name in names:
        lang = get_lang(name)
        if lang is None:
            continue
        hotel.name.add(name["#text"], lang=lang)
    phones = wrap_into_list(item.get("phone") or [])
    for phone in phones:
        if phone.get("number"):
            hotel.phone.add(phone["number"], type=phone_map.get(phone["type"]))

    hotel.original_id = item["company-id"]
    if "coordinates" in item:
        hotel.lat = item["coordinates"]["lat"]
        hotel.lon = item["coordinates"]["lon"]
    if "url" in item:
        hotel.url.add(item["url"], type=enums.UrlType.MAIN)
    if "add-url" in item:
        hotel.url.add(item["add-url"], type=enums.UrlType.BOOKING)
    hotel.rubric = enums.HotelRubric(item["rubric-id"])

    address = defaultdict(dict)
    country_code = None
    countries = wrap_into_list(item["country"] or [])
    for country in countries:
        lang = get_lang(country)
        country_name = country.get("#text")
        if lang is None or country_name is None:
            continue
        address[lang]["country"] = country_name
        country_code = country_code or country_name_to_code.get(country_name)

    localities = wrap_into_list(item["locality-name"] or [])
    for locality in localities:
        lang = get_lang(locality)
        if lang is None or locality.get("#text") is None:
            continue
        city = locality["#text"]
        address[lang]["city"] = city
        if city is not None:
            hotel._city.add(city, lang=lang)

    address_key = "address" if "address" in item else "street"

    if address_key in item:
        addresses = item[address_key]
        if not isinstance(addresses, list):
            addresses = [addresses]
        for addr in addresses:
            if "#text" not in addr:
                continue
            lang = get_lang(addr)
            if lang is None:
                continue
            if lang not in address:
                address[lang] = {}
            address[lang]["address"] = addr["#text"]
    else:
        warnings.warn(GeoWarning("Hotel has no street address"))

    # todo: refactor this.
    num_good_addresses = sum([addr.get("address") is not None and len(addr.get('address').strip()) > 0 for addr in six.itervalues(address)])
    for lang, addr in six.iteritems(address):
        if num_good_addresses == 0 or (addr.get('address') is not None and len(addr.get('address').strip()) > 0):  # no use in address without address
            addr_list = [a for a in [addr.get("country"), addr.get("city"), addr.get("address")] if a is not None]
            hotel.address.add(", ".join(addr_list), lang=lang)
    if num_good_addresses == 0:
        warnings.warn(GeoWarning("Hotel has empty street address"))
    if country_code is None:
        warnings.warn(GeoWarning("Country not recognized - '{}'".format(address[enums.Language.RU].get("country"))))
        return None
    hotel.country = country_code

    hotel.actualization_time = item.get("actualization-date")

    # photos
    photos = item.get("photos", {})
    gallery_url = {"gallery_url": photos["@gallery-url"]} if "@gallery-url" in photos else {}
    photos = photos.get("photo", [])
    if isinstance(photos, dict):
        photos = [photos]
    for photo in photos:
        hotel.photos.add(link=photo["@url"], **gallery_url)

    email = item.get('email')
    if email is not None:
        if type(email) in [str, unicode]:
            hotel.email = [e.strip() for e in email.split(',')]
        else:
            hotel.email = email
    hotel.zip_index = item.get('post-index')

    for feature in wrap_into_list(item.get("feature-boolean", [])):
        try:
            hotel.__setattr__(feature['@name'], bool(feature['@value']))
        except Exception as e:
            if debug:
                warnings.warn(AmenityWarning("Unable to set boolean feature \"{}\"to value \"{}\". Error: {}".format(feature['@name'], bool(feature['@value']), e)))

    for feature in wrap_into_list(item.get("feature-enum-multiple", [])) + wrap_into_list(item.get("feature-enum-single", [])):
        name, value = feature['@name'], feature['@value']
        try:
            enum_class_name = name.title().replace('_', '')
            enum_class = getattr(features_enums, enum_class_name)
            enum_value = enum_class.__getattr__(value)
            hotel.__setattr__(name, enum_value)
        except Exception as e:
            if name == 'tv':
                hotel.tv_in_room = True
                continue
            if name == 'internet_in_hotel':
                cont = False
                if 'free' in value:
                    hotel.internet_in_hotel = features_enums.InternetInHotel.free_internet
                    cont = True
                if 'paid' in value:
                    hotel.internet_in_hotel = features_enums.InternetInHotel.paid_internet
                    cont = True
                if 'wi_fi' in value or 'wifi' in value:
                    hotel.wi_fi = True
                    cont = True
                if cont:
                    continue
            if debug:
                warnings.warn(AmenityWarning("Unable to set enum feature \"{}\" to value \"{}\". Error: {}".format(feature['@name'], feature['@value'], e)))

    for feature in wrap_into_list(item.get("feature-numeric-single", [])):
        try:
            hotel.__setattr__(feature['@name'], int(feature['@value']))
        except Exception as e:
            if debug:
                warnings.warn(AmenityWarning("Unable to set numeric feature \"{}\" to value \"{}\". Error: {}".format(feature['@name'], feature['@value'], e)))

    for feature in wrap_into_list(item.get("feature-range-in-units-single", [])):
        try:
            if feature.get("@name") == "price_room":
                hotel._min_rate = feature.get("@from")
                hotel._currency = feature.get("@unit-value")
            else:
                warnings.warn(AmenityWarning("Unknown feature {}".format(feature.get("@name"))))
        except Exception as e:
            if debug:
                warnings.warn(AmenityWarning("Unable to set numeric feature \"{}\" to value \"{}\". Error: {}".format(feature['@name'], feature['@value'], e)))

    add_info = {rec['name']: rec['value'] for rec in wrap_into_list(item.get('add-info', {}).get('param', []))}
    hotel._ref_booking_id = add_info.get('BookingID')
    hotel._ref_expedia_id = add_info.get('ExpediaID')
    rating = add_info.get('Rating')
    hotel._review_rating = float(rating) if rating else None

    hotel._hotelscombined_key = add_info.get('HotelKey')

    descriptions = wrap_into_list(item.get("description", []))
    for description in descriptions:
        if description.get('#text'):
            hotel._description.add(description.get('#text'), lang=get_lang(description))
        elif debug:
            warnings.warn(DictionaryWarning("Bad 'description' value - key '#text' missing"))
    return hotel
