# coding=utf-8
import logging
import string
from contextlib import contextmanager

RUSSIAN_LETTERS = u"абвгдеёжзийклмнопрстуфхцчшщъыьэюяАБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ"


def format_to_text(s):
    try:
        return "{}".format(s)
    except:
        dec = None
        try:
            dec = "{}".format(s.decode('utf-8'))
        except:
            pass
        enc = None
        try:
            enc = "{}".format(s.encode('utf-8'))
        except:
            pass
        if dec and enc:
            return "Encoded: {} Decoded: {}".format(enc, dec)
        elif dec or enc:
            return dec or enc
        else:
            return "[undisplayable-value]"


def to_unicode(s):
    return s.decode('utf-8') if s is not None and not isinstance(s, unicode) else s


def to_utf8(s):
    return s.encode('utf-8') if s is not None and isinstance(s, unicode) else s


def wrap_into_list(l):
    return [l, ] if not isinstance(l, (list, tuple)) else l


def extract_value_from_list(l):
    return l[0] if type(l) is list and len(l) > 0 else l


def determine_language(s):
    if s is None:
        return None
    s = to_unicode(s)
    letters = [c for c in s if not c.isdigit() and not c.isspace() and c not in string.punctuation]
    if not letters:  # letters = []
        return None
    if sum(c not in string.ascii_letters for c in letters) < 2:  # house number sometimes has 'A' in ru...
        return 'EN'  # en - only en letters
    if sum(c not in RUSSIAN_LETTERS for c in letters) < 2:  # house number sometimes has 'A' in en...
        return 'RU'  # ru - only ru letters
    if all(c in (RUSSIAN_LETTERS + string.ascii_letters) for c in letters):
        return 'mixed'  # mixed - both ru and en letters
    return 'other'  # other - some other language with unknown characters


@contextmanager
def all_logging_disabled(highest_level=logging.CRITICAL):
    """
    A context manager that will prevent any logging messages
    triggered during the body from being processed.
    :param highest_level: the maximum logging level in use.
      This would only need to be changed if a custom level greater than CRITICAL
      is defined.
    """
    # two kind-of hacks here:
    #    * can't get the highest logging level in effect => delegate to the user
    #    * can't get the current module-level override => use an undocumented
    #       (but non-private!) interface

    previous_level = logging.root.manager.disable

    logging.disable(highest_level)

    try:
        yield
    finally:
        logging.disable(previous_level)
