import logging
import time
from contextlib import contextmanager
from travel.library.python.solomon_push_client import SolomonPushClient
from travel.hotels.lib.python import ytlib


LOG = logging.getLogger(__name__)


class NullClient(object):
    def send(self, sensor, value, **labels):
        pass

    def upload(self):
        pass


class YtSender(object):
    def __init__(self, yt_path):
        self.data = []
        self.yt_path = ytlib.join(yt_path, "@metrics")

    def send(self, sensor, value, **labels):
        labels['sensor'] = sensor
        labels['value'] = value
        self.data.append(labels)

    def upload(self):
        ytlib.yt.set(self.yt_path, self.data)


class Metrics(object):
    def __init__(self, partner, timelabel, yt_path, args):
        if args.send_metrics_to_solomon:
            self.solomon_sender = SolomonPushClient(
                project=args.solomon_project,
                cluster=args.solomon_cluster,
                service=args.solomon_service,
                token=args.solomon_token,
                ts=timelabel,
                labels={'partner': partner},
            )
        else:
            self.solomon_sender = NullClient()
        self.yt_sender = YtSender(yt_path)

    def send(self, sensor, value, **labels):
        LOG.info("Collected metric: '%s', %s, %s", sensor, value, labels)
        self.solomon_sender.send(sensor, value, **labels)
        self.yt_sender.send(sensor, value, **labels)

    @contextmanager
    def batch_send(self):
        try:
            yield
        finally:
            self.upload()

    @contextmanager
    def timed(self, name):
        started = time.time()
        try:
            yield
        finally:
            delta = time.time() - started
            delta_min = int(delta / 60)
            delta_sec = delta - delta_min * 60
            LOG.info("Block '{}' took {}m {:.2f}s".format(name, delta_min, delta_sec))
            name += '_time'
            self.send('processing_time', delta, tag=name)

    def upload(self):
        [sender.upload() for sender in (self.solomon_sender, self.yt_sender)]
