import enum
import six

from travel.hotels.feeders.lib.model.enums import Language, HotelRubric, PageType, UrlType

DEFAULT_FIELD_PRIORITY = 999
PRIVATE_FIELD_PRIORITY = 999999


def compose_error_message(message, field, value, target_type=None):
    error_message = "{message}\nfield: {field}\nvalue: {value}".format(message=message, field=field, value=repr(value))
    if target_type is not None:
        error_message += "\ntarget_type: {target_type}".format(target_type=target_type)
    return error_message


class Simple(object):
    def __init__(self, name, allow_multi, type_class=None, enum_as_value=False):
        self.name = name
        self.allow_multi = allow_multi
        self.values = [] if self.allow_multi else None
        self.type_class = type_class
        self.enum_as_value = enum_as_value

    def output(self):
        if self.allow_multi and not self.values:  # if it's a list and it's empty
            return self.name, None
        else:
            return self.name, self.values

    def __repr__(self):
        if self.allow_multi:  # if it's a list
            if not self.values:  # if it's empty
                return str(None)
            if len(self.values) == 1:  # if only 1 value
                return str(self.values[0])
        return str(self.values)

    def cast(self, value):
        if self.type_class is not None and value is not None:
            if not isinstance(value, self.type_class):
                raise ValueError(compose_error_message(
                    message="Value is of incorrect type",
                    value=value,
                    field=self.name,
                    target_type=self.type_class.__name__
                ))
        if value is not None and isinstance(value, enum.Enum):
            if self.enum_as_value:
                return value.value
            else:
                return value.name
        return value

    def set(self, value):
        value = self.cast(value)
        if self.allow_multi:
            self.values = [value]
        else:
            self.values = value

    def add(self, value):
        value = self.cast(value)
        if self.allow_multi:
            self.values.append(value)
        elif self.values is None:
            self.values = value
        else:
            raise ValueError(compose_error_message(
                message="Field may not have more than one value",
                value=value,
                field=self.name,
            ))


class Float(Simple):
    def cast(self, value):
        return float(value) if value is not None else None


class Int(Simple):
    def cast(self, value):
        return int(value) if value is not None else None


class String(Simple):
    def cast(self, value):
        return str(value) if value is not None else None


class Dict(Simple):
    allowed_keys = None
    required_keys = {"value"}

    def __init__(self, name, allow_multi, value_key="value"):
        super(Dict, self).__init__(name, allow_multi)
        self.value_key = value_key

    def cast(self, value):
        if isinstance(value, dict):
            for k, v in six.iteritems(value):
                if self.allowed_keys is not None and k not in self.allowed_keys:
                    raise ValueError(compose_error_message(
                        message="Unexpected dictionary key '{key}'".format(key=k),
                        value=value,
                        field=self.name
                    ))
                if v is not None:
                    value[k] = self.cast_dict_item(k, v)
            if self.required_keys is not None:
                for rk in self.required_keys:
                    if rk not in value.keys():
                        raise ValueError(compose_error_message(
                            message="Missing required key '{key}'".format(key=rk),
                            value=value,
                            field=self.name
                        ))
        elif value is not None:
            value = self.cast_dict_item("value", value)
        return value

    def cast_dict_item(self, key, value):
        return value

    def set(self, value=None, **kwargs):
        if value is not None:
            kwargs["value"] = value
        super(Dict, self).set(self.build(kwargs))

    @staticmethod
    def build(kwargs):
        if not kwargs:
            val = None
        else:
            val = {}
            for k, v in six.iteritems(kwargs):
                val[k] = v
        return val

    def add(self, value=None, **kwargs):
        if value is not None:
            kwargs["value"] = value
        super(Dict, self).add(self.build(kwargs))

    def output(self):
        name, value = super(Dict, self).output()
        if isinstance(value, list):
            val_list = value
        else:
            val_list = [value]
        for v in val_list:
            if v is not None and "value" in v:
                v[self.value_key] = v.pop("value")
        return name, value


class Localized(Dict):
    allowed_keys = {"value", "lang"}

    def cast_dict_item(self, key, value):
        if key == "lang":
            if not isinstance(value, Language):
                raise ValueError(compose_error_message(
                    message="'lang' should be a valid Language, current type: {}".format(type(value)),
                    value=value,
                    field=self.name
                ))
            return Language(value).name
        return value


class TypedLocalized(Localized):
    allowed_keys = {"value", "lang", "type"}

    def __init__(self, name, allow_multi, type_key="type", type_class=None):
        super(TypedLocalized, self).__init__(name, allow_multi)
        self.type_class = type_class
        self.type_key = type_key

    def cast_dict_item(self, key, value):
        if key == "type":
            if self.type_class is not None and not isinstance(value, self.type_class):
                raise ValueError(compose_error_message(
                    message="'type' is of incorrect type: {}".format(type(value)),
                    value=value,
                    field=self.name,
                    target_type=self.type_class.__name__
                ))
            if isinstance(value, enum.Enum):
                return value.name
            else:
                return value
        return super(TypedLocalized, self).cast_dict_item(key, value)

    def output(self):
        name, value = super(TypedLocalized, self).output()
        if isinstance(value, list):
            val_list = value
        else:
            val_list = [value]
        for v in val_list:
            if isinstance(v, dict) and "type" in v:
                v[self.type_key] = v.pop("type")
        return name, value


class Rubric(Dict):
    allowed_keys = {"value", "priority"}
    required_keys = {"value"}

    def cast_dict_item(self, key, value):
        if key == "value":
            if not isinstance(value, HotelRubric):
                raise ValueError(compose_error_message(
                    message="Value is of incorrect type: {}".format(type(value)),
                    value=value,
                    field=self.name,
                    target_type="HotelRubric"
                ))
            return value.value
        return value


class Url(Dict):
    allowed_keys = {"value", "type", "page_type"}

    def cast_dict_item(self, key, value):
        if key == "type":
            if not isinstance(value, UrlType):
                raise ValueError(compose_error_message(
                    message="'type' is of incorrect type: {}".format(type(value)),
                    value=value,
                    field=self.name,
                    target_type="UrlType"
                ))
        if key == "page_type":
            if not isinstance(value, PageType):
                raise ValueError(compose_error_message(
                    message="'page_type' has incorrect type",
                    value=value,
                    field=self.name,
                    target_type="PageType"
                ))
        if isinstance(value, enum.Enum):
            return value.name
        else:
            return value


class Feature(Dict):
    def __init__(self, name, allow_multi, feature_name=None, type_class=None, value_key="value"):
        super(Feature, self).__init__(name, allow_multi, value_key)
        self.feature_name = feature_name
        self.type_class = type_class

    def cast_dict_item(self, key, value):
        if key == "value":
            if self.type_class is not None:
                if not isinstance(value, self.type_class):
                    raise ValueError(compose_error_message(
                        message="Value is of incorrect type",
                        value=value,
                        field=self.name,
                        target_type=self.type_class.__name__
                    ))
                if isinstance(value, enum.Enum):
                    return value.name
                else:
                    return value
        return super(Feature, self).cast_dict_item(key, value)

    def output(self):
        name, val = super(Feature, self).output()
        if not isinstance(val, list):
            val = [val]
        res = []
        for v in val:
            if v is not None:
                v["id"] = self.feature_name
            res.append(v)
        if len(res) == 1:
            res = res[0]
        return name, res


class EnumFeature(Feature):
    def __init__(self, name, allow_multi, feature_name=None, type_class=None):
        super(EnumFeature, self).__init__(name, allow_multi, feature_name, type_class, "enum_id")


class MinMaxFeature(Feature):
    def __init__(self, name, allow_multi, feature_name=None, type_class=None):
        super(MinMaxFeature, self).__init__(name, allow_multi, feature_name, type_class)

    def cast_dict_item(self, key, value):
        if key == "value":
            if self.type_class is not None:
                if not isinstance(value, self.type_class):
                    raise ValueError(compose_error_message(
                        message="Value is of incorrect type",
                        value=value,
                        field=self.name,
                        target_type=self.type_class.__name__
                    ))
            if not isinstance(value, tuple) or len(value) != 2:
                raise ValueError(compose_error_message(
                    message="Value should be a tuple of two elements (min, max)",
                    value=value,
                    field=self.name
                ))
            return super(MinMaxFeature, self).cast_dict_item(key, value)

    def output(self):
        name, val = super(MinMaxFeature, self).output()
        if val is not None and "value" in val:
            min, max = val.pop("value")
            val["min"] = min
            val["max"] = max
        return name, val


class Photos(Dict):
    allowed_keys = {"link", "gallery_url", "type", "tag", "custom_tags"}
    required_keys = {"link"}


class RoomType(Localized):
    required_keys = Localized.required_keys.union({
        "id",
    })
    allowed_keys = Localized.allowed_keys.union({
        "id",
        "description",
        "photos",
        "amenities",
        "area_square_meters",
        "max_allowed_occupancy",
        "expedia_bed_groups",
        "expedia_views",
        "travelline_kind",
        "bronevik_bed_sets",
    })


class Field(object):
    def __init__(self, name, field_type=Simple, allow_multi=False, schema_priority=None, **type_parameters):
        self.name = name
        self.field_type = field_type
        self.allow_multi = allow_multi
        if schema_priority is None:
            if self.name.startswith("_"):
                self.schema_priority = PRIVATE_FIELD_PRIORITY
            else:
                self.schema_priority = DEFAULT_FIELD_PRIORITY
        else:
            self.schema_priority = schema_priority
        self.type_parameters = type_parameters

    def __set__(self, instance, value):
        item = self.field_type(name=self.name, allow_multi=self.allow_multi, **self.type_parameters)
        item.set(value)
        if not hasattr(instance, "_field_dict"):
            setattr(instance, "_field_dict", {})
        instance._field_dict[self] = item

    def __get__(self, instance, owner):
        item = getattr(instance, "_field_dict", {}).get(self)
        if item is None:
            if not hasattr(instance, "_field_dict"):
                setattr(instance, "_field_dict", {})
            item = self.field_type(name=self.name, allow_multi=self.allow_multi, **self.type_parameters)
            instance._field_dict[self] = item
        return item

    def set(self, value=None, *args, **kwargs):
        # not actually used and is passed to underlying value instead
        # needed for better code highlight and completion
        pass

    def add(self, value=None, *args, **kwargs):
        # not actually used and is passed to underlying value instead
        # needed for better code highlight and completion
        pass


class FeatureField(Field):
    def __init__(self, name, type_class=None):
        super(FeatureField, self).__init__("feature", field_type=Feature, allow_multi=False, type_class=type_class,
                                           feature_name=name)


class BooleanFeatureField(FeatureField):
    def __init__(self, name):
        super(BooleanFeatureField, self).__init__(name, bool)


class EnumFeatureField(Field):
    def __init__(self, name, enum_type, allow_multi=False):
        super(EnumFeatureField, self).__init__("feature", field_type=EnumFeature, allow_multi=allow_multi,
                                               type_class=enum_type, feature_name=name)


class MinMaxFeatureField(Field):
    def __init__(self, name, type_class=None):
        super(MinMaxFeatureField, self).__init__("feature", field_type=MinMaxFeature, allow_multi=False,
                                                 type_class=type_class, feature_name=name)


class EnumValueFeature(Feature):
    def __init__(self, name, allow_multi, feature_name=None, type_class=None):
        super(EnumValueFeature, self).__init__(name, allow_multi, feature_name, type_class, "enum_id")

    def cast_dict_item(self, key, value):
        if key == "value":
            if self.type_class is not None:
                if isinstance(value, enum.Enum):
                    return value.value
        return super(EnumValueFeature, self).cast_dict_item(key, value)


class EnumValueFeatureField(Field):
    def __init__(self, name, enum_type, allow_multi=False):
        super(EnumValueFeatureField, self).__init__("feature", field_type=EnumValueFeature, allow_multi=allow_multi,
                                                    type_class=enum_type, feature_name=name)
