# coding=utf-8
import warnings

import six
from travel.hotels.feeders.lib import helpers
from travel.hotels.feeders.lib.model import enums
from travel.hotels.feeders.lib.model import fields, hotel_features

from travel.hotels.feeders.lib.model.log_message_types import DataCheckWarning, GeoWarning, LocalizationWarning
from travel.hotels.feeders.lib.model.log_message_types import NameWarning, PhoneWarning, DebugWarning

MIN_PHONE_DIGITS_COUNT = 4
MIN_NAME_CHARACTER_COUNT = 2


class Hotel(hotel_features.HotelFeatures):
    original_id = fields.Field("originalId", schema_priority=0)
    _partner = fields.Field("_partner")

    name = fields.Field("name", field_type=fields.TypedLocalized, schema_priority=1, allow_multi=True,
                        type_class=enums.NameType, type_key="name_type")

    address = fields.Field("address", field_type=fields.Localized, schema_priority=2, allow_multi=True,
                           value_key="one_line")
    address_add = fields.Field("addressAdd", field_type=fields.Localized, schema_priority=2, allow_multi=True)
    country = fields.Field("country", schema_priority=2)
    _city = fields.Field("_city", field_type=fields.Localized, allow_multi=True)
    lat = fields.Field("lat", field_type=fields.Float, schema_priority=2)
    lon = fields.Field("lon", field_type=fields.Float, schema_priority=2)
    zip_index = fields.Field("zipIndex", schema_priority=2)

    phone = fields.Field("phone", field_type=fields.TypedLocalized, schema_priority=3, allow_multi=True,
                         type_class=enums.PhoneType, type_key="type")
    url = fields.Field("url", field_type=fields.Url, schema_priority=3, allow_multi=True)
    email = fields.Field("email", schema_priority=3, allow_multi=True)

    photos = fields.Field("photos", field_type=fields.Photos, allow_multi=True)

    room_types = fields.Field("roomTypes", field_type=fields.RoomType, allow_multi=True)

    working_time = fields.Field("workingTime", field_type=fields.Localized, allow_multi=True)

    rating = fields.Field("rating", field_type=fields.String)
    _review_count = fields.Field("_reviewCount", field_type=fields.Int)
    _review_rating = fields.Field("_reviewRating", field_type=fields.Float)
    _ranking = fields.Field("_ranking", field_type=fields.Int)  # booking21 ranking and hotelscombined2 popularity
    _checkout_begin_time = fields.Field("_checkoutBeginTime")
    _checkout_end_time = fields.Field("_checkoutEndTime")
    _checkin_begin_time = fields.Field("_checkinBeginTime")
    _checkin_end_time = fields.Field("_checkinEndTime")

    _booking_review_total_rating = fields.Field("_bookingReviewTotalRating", field_type=fields.Float)
    _booking_review_staff_rating = fields.Field("_bookingReviewStaffRating", field_type=fields.Float)
    _booking_review_services_rating = fields.Field("_bookingReviewServicesRating", field_type=fields.Float)
    _booking_review_clean_rating = fields.Field("_bookingReviewCleanRating", field_type=fields.Float)
    _booking_review_comfort_rating = fields.Field("_bookingReviewComfortRating", field_type=fields.Float)
    _booking_review_location_rating = fields.Field("_bookingReviewLocationRating", field_type=fields.Float)
    _booking_review_value_rating = fields.Field("_bookingReviewValueRating", field_type=fields.Float)
    _booking_review_count = fields.Field("_bookingReviewCount", field_type=fields.Int)

    _expedia_onsite_payment_currency = fields.Field("_expediaOnsitePaymentCurrency")
    _expedia_airports_preferred_code = fields.Field("_expediaAirportsPreferredCode")
    _expedia_descriptions_rooms = fields.Field("_expediaDescriptionsRooms", field_type=fields.Localized, allow_multi=True)
    _expedia_descriptions_dining = fields.Field("_expediaDescriptionsDining", field_type=fields.Localized, allow_multi=True)
    _expedia_descriptions_location = fields.Field("_expediaDescriptionsLocation", field_type=fields.Localized, allow_multi=True)
    _expedia_descriptions_attractions = fields.Field("_expediaDescriptionsAttractions", field_type=fields.Localized, allow_multi=True)
    _expedia_descriptions_renovations = fields.Field("_expediaDescriptionsRenovations", field_type=fields.Localized, allow_multi=True)
    _expedia_descriptions_business_amenities = fields.Field("_expediaDescriptionsBusinessAmenities", field_type=fields.Localized, allow_multi=True)
    _expedia_descriptions_amenities = fields.Field("_expediaDescriptionsAmenities", field_type=fields.Localized, allow_multi=True)
    _expedia_checkin_min_age = fields.Field("_expediaCheckinMinAge", field_type=fields.Int)
    _expedia_property_pansion = fields.Field("_expediaPropertyPansion", field_type=fields.String)

    _hotelscombined_key = fields.Field("_hotelscombinedKey")
    _dolphin_region = fields.Field("_dolphinRegion")
    _dolphin_city = fields.Field("_dolphinCity")
    _dolphin_district = fields.Field("_dolphinDistrict")
    _dolphin_original_name = fields.Field("_dolphinOriginalName")
    _dolphin_original_address = fields.Field("_dolphinOriginalAddress")
    _travelline_enabled_for_yandex = fields.Field("_travellineEnabledForYandex", field_type=fields.Int)
    _travelline_max_los = fields.Field("_travellineMaxLOS", field_type=fields.Int)
    _bnovo_uid = fields.Field("_bnovoUid")

    _description = fields.Field("_description", field_type=fields.Localized, allow_multi=True)
    _min_rate = fields.Field("_minRate", field_type=fields.Float)
    _currency = fields.Field("_currency")
    _ostrovok_policies = fields.Field("_ostrovokPolicies", field_type=fields.Localized, allow_multi=True)
    _ref_booking_id = fields.Field("_refBookingId", field_type=fields.Int)
    _ref_expedia_id = fields.Field("_refExpediaId", field_type=fields.Int)
    chain_id = fields.Field("chainId", allow_multi=True)
    _chain = fields.Field("_chain")
    actualization_time = fields.Field("actualizationTime")

    rubric = fields.Field("rubric", field_type=fields.Rubric, allow_multi=True)
    publishing_status = fields.Field("publishingStatus", type_class=enums.PublishingStatus)

    # scalar features
    wifi_name = fields.FeatureField("ssid")
    # range features
    beds_per_room = fields.MinMaxFeatureField("the_number_of_places_in_the_room", type_class=int)
    # в фидах не будет. Зарезервированно под контур свежести
    price_room_day = fields.MinMaxFeatureField("price_room", type_class=float)
    price_room_hour = fields.MinMaxFeatureField("price_room_hour", type_class=float)
    price_bed = fields.MinMaxFeatureField("price_bed", type_class=float)

    label_hash = None
    label_proto = None

    def to_dict(self, partner_name, publish_features=True, allowed_fields=None, hidden_fields=None, list_to_string_fields=None):
        return serialize(self, partner_name=partner_name, publish_features=publish_features, allowed_fields=allowed_fields, hidden_fields=hidden_fields, list_to_string_fields=list_to_string_fields)

    @classmethod
    def get_yt_schema(cls, partner_name, publish_features=True, allowed_fields=None, hidden_fields=None):
        return build_schema(cls, partner_name=partner_name, publish_features=publish_features, allowed_fields=allowed_fields, hidden_fields=hidden_fields)


def get_full_dict(cls):
    res = {}
    for subcls in cls.__mro__:
        if subcls.__name__ != "object":
            res.update(subcls.__dict__)
    return res


def serialize(obj, partner_name=None, publish_features=True, allowed_fields=None, hidden_fields=None, list_to_string_fields=None):
    if allowed_fields is None:
        allowed_fields = []
    if hidden_fields is None:
        hidden_fields = []
    if list_to_string_fields is None:
        list_to_string_fields = []
    res = {}
    for attr_name, attr in six.iteritems(get_full_dict(obj.__class__)):
        if isinstance(attr, fields.Field):
            name, value = getattr(obj, attr_name).output()
            if name not in allowed_fields:
                if name in hidden_fields or (name.startswith('_') and (partner_name is None or not name.startswith('_' + partner_name))):
                    if value is not None and value != []:
                        warnings.warn(DataCheckWarning("Not empty value '{}' for hidden field {}".format(value, name)))
                    continue
            if name not in res:
                res[name] = value
            else:
                if not isinstance(res[name], list):
                    if res[name] is None:
                        res[name] = []
                    else:
                        res[name] = [res[name]]
                if isinstance(value, list):
                    res[name].extend(value)
                else:
                    if value is not None:
                        res[name].append(value)
    for field in list_to_string_fields:
        if field in res and res[field] is not None:  # Spravochnik doesn't accept list of emails. pass as ' '.join(list)
            data = res[field]
            if all([type(e) is list for e in data]):
                data = set().union(*data)
            if not all([type(e) in [str, unicode] for e in data]):
                data = [repr(x) for x in data]
                warnings.warn(DataCheckWarning("Data {} should contain only strings".format(field)))
            res[field] = ' '.join(set(data))

    if not publish_features and 'feature' in res:
        allowed_features = ['star', 'hotel_type']
        res['_feature'] = [feature for feature in res['feature'] if feature['id'] not in allowed_features]
        res['feature'] = [feature for feature in res['feature'] if feature['id'] in allowed_features]

    return res


def build_schema(cls, partner_name=None, publish_features=True, allowed_fields=None, hidden_fields=None):
    if allowed_fields is None:
        allowed_fields = []
    if hidden_fields is None:
        hidden_fields = []

    schema = []
    has_feature = False
    for _, f in six.iteritems(get_full_dict(cls)):
        if isinstance(f, fields.Field):
            c_name = f.name
            if c_name not in allowed_fields:
                if c_name in hidden_fields:
                    continue
                if c_name.startswith('_') and (partner_name is None or not c_name.startswith('_' + partner_name)):
                    if c_name.startswith('_hotels101'):
                        raise Exception("Okay, seems it's time to rename 101hotels into hotels101. or add the field {} in hotels101's allowed_fields list".format(c_name))
                    continue
            c_priority = f.schema_priority
            if issubclass(f.field_type, fields.Feature):
                has_feature = True
                continue
            elif issubclass(f.field_type, fields.Float):
                c_type = "double"
            elif issubclass(f.field_type, fields.Int):
                c_type = "int64"
            else:
                if f.allow_multi:
                    c_type = "any"
                else:
                    c_type = "string"
            schema.append(dict(name=c_name, type=c_type, priority=c_priority))
    if has_feature:
        schema.append(dict(name="feature", type="any", priority=fields.DEFAULT_FIELD_PRIORITY))
        if not publish_features:
            schema.append(dict(name="_feature", type="any", priority=fields.DEFAULT_FIELD_PRIORITY))
    schema = sorted(schema, key=lambda item: (item.get("priority"), item.get("name")))
    schema = list(schema)
    for item in schema:
        del item["priority"]
    return schema


def check_data(record, debug=False):
    name = record.name.values
    if ''.join(a['value'] for a in name) == '':
        warnings.warn(NameWarning("Missing name"))
    elif all([len(a['value']) < MIN_NAME_CHARACTER_COUNT for a in name]):
        warnings.warn(NameWarning("All names too short: less than {} characters".format(MIN_NAME_CHARACTER_COUNT)))
    elif any([len(a['value']) < MIN_NAME_CHARACTER_COUNT for a in name]):
        warnings.warn(NameWarning("Too short name translation: less than {} characters".format(MIN_NAME_CHARACTER_COUNT)))
    country = record.country.values
    if country is None:
        warnings.warn(GeoWarning("Missing country"))
    lat = record.lat.values
    lon = record.lon.values
    if lon is None:
        warnings.warn(GeoWarning("Missing longitude"))
    elif abs(lon) > 180:
        warnings.warn(GeoWarning("abs(Longitude) is greater than 180"))
    if lat is None:
        warnings.warn(GeoWarning("Missing latitude"))
    elif abs(lat) > 90:
        warnings.warn(GeoWarning("abs(Latitude) is greater than 90; swapping lon and lat"))
        record.lon, record.lat = lat, lon

    phones = record.phone.values
    to_delete = []
    if not isinstance(phones, list):
        phones = [phones]
    for phone in phones:
        if phone is not None and ('value' in phone) and (phone['value'] is not None):
            if len([i for i in phone['value'] if i.isdigit()]) < MIN_PHONE_DIGITS_COUNT:
                warnings.warn(PhoneWarning("Phone is too short: less than {} digits".format(MIN_PHONE_DIGITS_COUNT)))
                to_delete.append(phone)
    for phone in to_delete:
        del record.phone.values[record.phone.values.index(phone)]

    for address in record.address.values:
        addr = helpers.to_unicode(address['value'])
        if addr == '':
            warnings.warn(GeoWarning("Empty address value"))
            continue
        lang = helpers.determine_language(addr)
        declared_lang = address.get('lang')
        if lang == 'mixed':  # definitely bad
            message = "Address has mixed language"
            if debug:
                message += ": {}".format(helpers.format_to_text(addr))
            warnings.warn(LocalizationWarning(message))
        if {lang, declared_lang} == {'RU', 'EN'}:
            message = "Address language is ru but en"
            if debug:
                message += ": {}".format(helpers.format_to_text(addr))
            warnings.warn(LocalizationWarning(message))
        if debug and lang != declared_lang and declared_lang not in {'EN', None} and lang not in {'mixed', 'other', 'EN'}:
            # known reasonable cases: ES and EN, ES and other,
            warnings.warn(DebugWarning("Mismatch in address lang: {} and {}, address: {}".format(declared_lang, lang, helpers.format_to_text(addr))))
    for attr_name, attr in six.iteritems(record.__class__.__dict__):
        if isinstance(attr, fields.Field):
            values = getattr(record, attr_name).values
            if values is None:
                continue
            if issubclass(attr.field_type, fields.Localized):
                if not isinstance(values, list):
                    values = [values]
                for value in values:
                    if value is not None and 'lang' in value:
                        lang = value['lang']
                        if lang in [None, 'None', 'Null']:
                            warnings.warn(LocalizationWarning("Localized field has invalid language '{}'".format(lang)))
