from __future__ import print_function, unicode_literals

from argparse import ArgumentParser

import logging
import requests


def check_status(status_url, provider_id, feed_id):
    params = {'provider_id': provider_id, 'feed_id': feed_id}
    r = requests.get(status_url, params=params)
    return r.json()['loading_status'], r.url


def send_notification(message, color, notification_url):
    attachments = {
        'attachments': [
            {
                'fallback': message,
                'text': message,
                'color': color,
            },
        ],
    }
    requests.post(notification_url, json=attachments)


def get_options():
    default_status_url = 'https://feeds.altay-api.search.yandex.net/feed/status'
    default_notification_url = 'http://travel-slack-forwarder.yandex.net:9098/travel-utils'
    parser = ArgumentParser()
    parser.add_argument('--status-url', default=default_status_url)
    parser.add_argument('--provider-id', default='3501497179')
    parser.add_argument('--feed-id', default='1')
    parser.add_argument('--notification-url', default=default_notification_url)
    parser.add_argument('--always-notify', action='store_true')
    return parser.parse_args()


def main():
    logging.basicConfig(format='%(asctime)s\t%(levelname)s\t%(message)s', level=logging.INFO)
    options = get_options()
    logging.info('working with "%r"', options)
    status, url = check_status(options.status_url, options.provider_id, options.feed_id)
    logging.info('got status "%s"', status)
    if options.always_notify or status != 'Ok':
        message = 'Feeders altay import status: `{}`\nDetails at {}\n<!channel>'.format(status, url)
        color = 'good' if status == 'Ok' else 'danger'
        send_notification(message, color, options.notification_url)
        logging.info('notification sent')
    logging.info('check finished')


if __name__ == '__main__':
    main()
