#!/usr/bin/python
# coding=utf-8

from __future__ import print_function, unicode_literals

from argparse import ArgumentParser
from datetime import date
from library.python import resource
from openpyxl import Workbook
from startrek_client import Startrek
from yql.api.v1.client import YqlClient
from yt.wrapper import ypath_join

import yt.wrapper as yt

import logging
import os


class Doc(object):
    DOC_STRUCTURE = []

    def __init__(self, data):
        self.data = self.filter(data)
        self.has_data = all((bool(d) for _, d in self.data))

    def filter(self, data):
        result = []
        for page in self.DOC_STRUCTURE:
            if 'include_warnings' in page:
                page_data = filter(lambda x: x['type'] in page['include_warnings'], data)
            elif 'exclude_warnings' in page:
                page_data = filter(lambda x: x['type'] not in page['exclude_warnings'], data)
            else:
                page_data = data
            result.append((page['sheet_name'], page_data))
        return result


class FeedersDoc(Doc):
    DOC_STRUCTURE = [
        {
            'sheet_name': 'Amenities',
            'include_warnings': {
                'AmenityWarning',
            },
        },
        {
            'sheet_name': 'Hotel types',
            'include_warnings': {
                'HotelTypeWarning',
            },
        },
        {
            'sheet_name': 'Rubrics',
            'include_warnings': {
                'RubricWarning',
            },
        },
    ]


class PartnersDoc(Doc):
    DOC_STRUCTURE = [
        {
            'sheet_name': 'Partners data',
            'exclude_warnings': {
                'AmenityWarning',
                'HotelTypeWarning',
                'RubricWarning',
            },
        },
    ]


PARTNER_TICKETS = {
    '101hotels': 'YATRAVELBD-130',
    'booking21': 'YATRAVELBD-133',
    'expedia': 'YATRAVELBD-134',
    'hotelscombined': 'YATRAVELBD-131',
    'ostrovok': 'YATRAVELBD-135',
    'travelline': 'YATRAVELBD-132',
}


CONFIG = {
    'feeders': {
        'doc': FeedersDoc,
        'tickets': {
            '*': 'HOTELS-3303',
        },
    },
    'partners': {
        'doc': PartnersDoc,
        'tickets': PARTNER_TICKETS,
    },
}


def exec_query(source, target, max_hotel_count, token_path):
    qt = resource.find("/query.template")
    q = qt % {'source': source, 'target': target, 'hotel_id_count': max_hotel_count}
    with YqlClient(token_path=token_path) as client:
        request = client.query(q)
        request.run()
        print(request)  # await workaround


def configure_yt(proxy, token_fn):
    if proxy is not None:
        if not proxy.endswith('.yt.yandex.net'):
            proxy = proxy + '.yt.yandex.net'
        yt.config["proxy"]["url"] = proxy
    if token_fn is not None:
        with open(token_fn) as f:
            token = f.read().strip()
        yt.config["token"] = token


def read_table(table_path):
    return list(yt.read_table(table_path))


def save_data(doc, fn):
    wb = Workbook(write_only=True)
    for sheet, data in doc.data:
        ws = wb.create_sheet(title=sheet)
        ws.append(['type', 'message', 'count', 'examples'])
        [ws.append([row['type'], row['message'], row['count_'], '\n'.join(row['example_hotel_ids'])]) for row in data]
    wb.save(fn)


def send_to_st(fn, ticket, token):
    client = Startrek(useragent='python', base_url='https://st-api.yandex-team.ru', token=token)
    issue = client.issues[ticket]
    issue.comments.create(text='Отчёт по логам фидеров', attachments=[fn])


def process_partner(partner, config, options):
    logging.info('processing "%s"', partner)

    source_table = ypath_join(options.path, partner, 'latest', 'parsed_aux', '_log')
    logging.info('source_table "%s"', source_table)
    dest_table = ypath_join(options.path, partner, 'latest', 'parsed_aux', '_stats')
    if not yt.exists(source_table):
        logging.info('path not found: "%s"', source_table)
        return
    logging.info('dest_table "%s"', dest_table)

    with yt.Transaction():
        if yt.exists(dest_table):
            yt.remove(dest_table)
    exec_query(source_table, dest_table, options.max_hotel_count, options.yql_token)
    logging.info('query completed')

    data = read_table(dest_table)
    logging.info('got table data')
    doc = config['doc'](data)
    if not doc.has_data:
        logging.info('nothing to save')
        return

    fn = '{}_{}_{}.xlsx'.format(options.config, partner, date.today())
    save_data(doc, fn)
    logging.info('data saved to "%s"', fn)

    ticket_map = config['tickets']
    ticket = ticket_map.get('*') or ticket_map[partner]
    with open(options.st_token) as f:
        st_token = f.read().strip()
    send_to_st(fn, ticket, st_token)
    logging.info('data sent to "%s"', ticket)

    os.remove(fn)
    logging.info('file "%s" removed', fn)


def get_options():
    default_path = '//home/travel/prod/feeds'
    configs = CONFIG.keys()
    parser = ArgumentParser()
    parser.add_argument('--st-token', default='st_token', help='token filename for ST')
    parser.add_argument('--yql-token', default='yql_token', help='token filename for YQL')
    parser.add_argument('--yt-token', default=None, help='token filename for YT')
    parser.add_argument('--yt-proxy', default='hahn.yt.yandex.net', help='YT proxy url')
    parser.add_argument('-p', '--path', default=default_path, help='path to feeds root')
    parser.add_argument('-c', '--max-hotel-count', default=10, help='max example hotel id count')
    parser.add_argument('--config', required=True, choices=configs)
    parser.add_argument('-t', '--force-ticket', help='use this ticket for any config')
    return parser.parse_args()


def main():
    logging.basicConfig(format='%(asctime)s\t%(levelname)s\t%(message)s', level=logging.INFO)
    options = get_options()
    logging.info('working with "%r"', options)
    config = CONFIG[options.config]
    if options.force_ticket is not None:
        config['tickets'] = {'*': options.force_ticket}
    configure_yt(options.yt_proxy, options.yt_token)
    partners = yt.list(options.path)
    for partner in partners:
        process_partner(partner, config, options)


if __name__ == '__main__':
    main()
