# coding=utf-8
import csv
import json
import logging
import requests
import warnings

from library.python import resource
from requests import HTTPError
from six import iteritems, text_type

from travel.hotels.feeders.lib import base, downloaders, parsers, helpers
from travel.hotels.feeders.lib.model import objects, enums
# Для всех ворнингов нужно использовать один из следующих типов (можно добавлять)
from travel.hotels.feeders.lib.model.log_message_types import ClosedHotelWarning, CoordinatesWarning

LOG = logging.getLogger(__name__)


class BNovo(base.Partner):
    name = "bnovo"

    min_records_count_accepted = None  # минимальное количество отелей при котором фид считается успешно построенным
    max_records_added_or_removed_count = None  # максимальное удаленное или добавленное количество отелей
    warn_records_count_change = None
    allowed_fields = base.Partner.allowed_fields + [
        '_bnovoUid'
    ]
    merge_translations = False

    bnovo_public_url = "https://public-api.reservationsteps.ru/v1/api"
    public_accounts_url = bnovo_public_url + "/accounts"
    public_room_types_url = bnovo_public_url + "/roomtypes"

    bnovo_private_url = "https://api.reservationsteps.ru/v1/api"
    private_auth_url = bnovo_private_url + "/auth"
    private_accounts_url = bnovo_private_url + "/accounts"

    retry_attempts = 5

    csv_based_hotels = []

    def load_hotels_csv(self):
        hotels_string = resource.find("/hotels.csv").splitlines()
        reader = UnicodeDictReader(hotels_string, 'utf-8', delimiter=';',
                                   fieldnames=['name', 'phone', 'email', 'city', 'address',
                                               'coordinates', 'point', 'timezone', 'uid'])
        i = 0
        for hotel in reader:
            self.csv_based_hotels.append(hotel)
            i += 1
            if self.limit is not None and i >= self.limit:
                break

        LOG.info("Total csv hotels count: {}".format(len(self.csv_based_hotels)))
        self.save_raw_feed([(hotel, {}) for hotel in self.csv_based_hotels], "hotels_csv")

    def authorize(self):
        LOG.info("authorizing in private API")
        resp = requests.post(self.private_auth_url, data=dict(username=self.login, password=self.password))
        if resp.status_code == 200:
            self.token = json.loads(resp.content).get("token")
        else:
            raise Exception("Unable to authorize in private api")

    def try_get_response(self, url, extra_info):
        last_error = None
        for _ in xrange(self.retry_attempts):
            try:
                with helpers.all_logging_disabled():
                    with self.download_feed(
                        url=url,
                        downloader=downloaders.StreamingDownloader(),
                        parser=parsers.DelimitedStreamParser(extra_info=extra_info),
                    ) as feed:
                        item, extra = next(feed)
                        return json.loads(item), extra
            except HTTPError as e:
                if e.response is not None and e.response.status_code == 204:
                    # no sense to retry 204's: there is no content
                    raise
                else:
                    LOG.warning("Error {} for url {}, will retry".format(str(e), url))
                    last_error = e
                    continue
        if last_error:
            raise last_error

    def try_get_hotel(self, hotel_obj):
        url = self.public_accounts_url + "?uid=" + hotel_obj["uid"]
        extra_info = dict(is_private=False, csv_data=hotel_obj)
        return self.try_get_response(url, extra_info)

    def try_get_rooms(self, account_id):
        url = "{}?account_id={}".format(self.public_room_types_url, account_id)
        extra_info = dict(is_private=False, csv_data=dict())
        rooms, _ = self.try_get_response(url, extra_info)
        return rooms

    def get_hotels_feed(self):
        failed_downloads = []
        loaded_hotels_count = 0
        index = 0
        self.authorize()
        private_uids = set()
        with helpers.all_logging_disabled():
            with self.download_feed(
                url="{}?token={}".format(self.private_accounts_url, self.token),
                downloader=downloaders.StreamingDownloader(),
                parser=parsers.JsonParser(
                    ijson_path="accounts.item",
                    extra_info=dict(is_private=True, csv_data={})
                )
            ) as private_feed:
                for hotel, extra in private_feed:
                    hotel_id = hotel["id"]
                    if hotel_id == 3056:  # тестовый яндекс-отель
                        continue
                    private_uids.add(hotel["uid"])
                    rooms = self.try_get_rooms(hotel_id)
                    hotel_info = dict(account=hotel)
                    hotel_info.update(rooms)
                    yield hotel_info, extra
                    loaded_hotels_count += 1
                    if self.limit is not None and loaded_hotels_count >= self.limit:
                        break

        for hotel_obj in self.csv_based_hotels:
            if hotel_obj["uid"] in private_uids:
                continue
            try:
                hotel_info, extra = self.try_get_hotel(hotel_obj)
                rooms = self.try_get_rooms(hotel_info["account"]["id"])
                hotel_info.update(rooms)
                yield hotel_info, extra
                loaded_hotels_count += 1
                if self.limit is not None and loaded_hotels_count >= self.limit:
                    break
            except HTTPError:
                failed_downloads.append(hotel_obj["uid"])
        index += 1
        LOG.info("Parsed %s JSON items; done!", loaded_hotels_count)
        LOG.warning("Failed download of following ids: {}".format(failed_downloads))

    def load_hotels(self):
        self.save_raw_feed(self.get_hotels_feed(), "hotels")

    def download_all_feeds(self):
        self.load_hotels_csv()
        self.load_hotels()

    def map(self, dict_item, info):
        return self.row_mapper.map(dict_item, info)

    def __init__(self, session, args):
        super(BNovo, self).__init__(session, args)
        self.row_mapper = BNovoRowMapper()
        self.login = args.login
        self.password = args.password
        self.token = None

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        arg_group = parser.add_argument_group(BNovo.name)
        arg_group.add_argument("--login", default='travel-hotels-partners@yandex-team.ru')
        arg_group.add_argument("--password", required=True)


class BNovoRowMapper(object):
    language_to_enum = {
        "ru": enums.Language.RU,
        "en": enums.Language.EN
    }
    name_hints = {
        "хостел": enums.HotelRubric.HOSTEL,
        "hostel": enums.HotelRubric.HOSTEL,
        "апартаменты": enums.HotelRubric.ACCOMMODATION_FOR_DAILY_RENT,
        "база отдыха": enums.HotelRubric.REST_HOUSE,
        "санаторий": enums.HotelRubric.RESORT,
        "загородный клуб": enums.HotelRubric.REST_HOUSE,
        "коттедж": enums.HotelRubric.ACCOMMODATION_FOR_DAILY_RENT,
        "оздоровительный": enums.HotelRubric.RESORT,
        "общежитие": enums.HotelRubric.DORMITORY,
        "туристическая база": enums.HotelRubric.DORMITORY,
        "турбаза": enums.HotelRubric.TOURIST_RESORT,
        "частн": enums.HotelRubric.ACCOMMODATION_FOR_DAILY_RENT,
    }

    def __init__(self):
        pass

    def map(self, dict_item, info):
        """
        Map dict structure to Hotel object
        :param dict_item: dict structure from partner
        :param info: meta information about request
        :return: Hotel object or None
        """
        csv_data = info["csv_data"]
        hotel_item = dict_item["account"]
        hotel = objects.Hotel()
        hotel._bnovo_uid = hotel_item["uid"]
        hotel.original_id = str(hotel_item["id"])
        if hotel_item["enabled"] != 1:
            warnings.warn(ClosedHotelWarning("Hotel is not enabled"))
            return None
        hotel.rubric = enums.HotelRubric.HOTEL
        name = hotel_item["name"]
        for hint, category in self.name_hints.iteritems():
            if helpers.to_unicode(hint) in helpers.to_unicode(name).lower():
                hotel.rubric = category
                break

        lang = hotel_item["language"]
        lang_enum = self.language_to_enum[lang]
        hotel.name.add(name, lang=lang_enum)
        if lang_enum == enums.Language.EN:
            address_string = helpers.to_unicode("Russia, ")
        else:
            address_string = helpers.to_unicode("Россия, ")
        address_string += helpers.to_unicode(hotel_item["address"])
        hotel.address.add(address_string, lang=lang_enum)

        hotel.email = hotel_item["email"]
        phone = hotel_item["phone"]
        if phone:
            hotel.phone.add(phone, type=enums.PhoneType.PHONE)
        geo_data = hotel_item["geo_data"]
        if geo_data == "(0,0)":
            if csv_data and csv_data["coordinates"] != "0":
                geo_data = csv_data["coordinates"]
            else:
                geo_data = None
        if geo_data is not None:
            lat, lon = geo_data[1:-1].split(',')
            hotel.lat = float(lat)
            hotel.lon = float(lon)
        else:
            warnings.warn(CoordinatesWarning("Missing coordinates"))
        hotel.country = "RU"

        for room in dict_item.get("rooms", list()):
            room_id = room["id"]
            name = room.get("name_ru")
            if not name:
                name = room["name"]
            description = room.get("description_ru")
            if not description:
                description = room.get("description")
            max_allowed_occupancy = {
                'adults': room.get("adults"),
                'children': room.get("children"),
            }

            photos = []
            raw_photos = room.get("photos")
            if raw_photos is None:
                raw_photos = list()
            for photo in raw_photos:
                link = dict(link=photo["url"])
                photos.append(link)
                hotel.photos.add(**link)

            hotel.room_types.add(
                id=room_id,
                lang=lang_enum,
                value=name,
                description=description,
                max_allowed_occupancy=max_allowed_occupancy,
                photos=photos,
            )

        return hotel


class UnicodeDictReader(csv.DictReader, object):
    def __init__(self, f, encoding, **kwargs):
        super(UnicodeDictReader, self).__init__(f, **kwargs)
        self.encoding = encoding

    @staticmethod
    def decode_value(item, encoding):
        if item is None:
            return item
        if isinstance(item, unicode):
            return item
        if isinstance(item, list):
            return [text_type(i, encoding) for i in item]
        return text_type(item, encoding)

    def next(self):
        row = super(UnicodeDictReader, self).next()
        encoding = self.encoding
        return {self.decode_value(key, encoding): self.decode_value(value, encoding) for key, value in iteritems(row)}
