# coding=utf-8
from gevent import monkey

#  gevent monkey patching must be performed before any other import

monkey.patch_all()

import gevent  # noqa
import six  # noqa
import logging  # noqa
import warnings  # noqa
import requests

from travel.hotels.feeders.lib import downloaders, parsers, base, helpers  # noqa
from travel.hotels.feeders.lib.model import objects, enums, features_enums  # noqa
from travel.hotels.feeders.lib.model.log_message_types import ClosedHotelWarning, HotelTypeWarning, RubricWarning, AmenityWarning, GeoWarning  # noqa
from travel.hotels.feeders.lib.model.log_message_types import LocalizationWarning, NameWarning  # noqa
from travel.hotels.feeders.partners.booking21 import features_mapping, hotel_type_mapping, rubrics_mapping  # noqa

LOG = logging.getLogger(__name__)


class Translations(dict):
    def __init__(self, desc):
        self["default"] = desc["name"]
        translations = desc.get("translations") or []
        if type(translations) not in (list, tuple):
            translations = [translations]
        for translation in translations:
            self[translation["language"]] = translation["name"]


class CityDescription(object):
    def __init__(self, desc):
        self.id = desc["city_id"]
        self.num_hotels = int(desc["nr_hotels"])
        self.translations = Translations(desc)


class CountryDescription(object):
    def __init__(self, desc):
        self.code = desc["country"]
        self.city_map = None
        self.num_hotels = None
        self.translations = Translations(desc)

    def set_cities_feed(self, cities_feed):
        self.city_map = {city["city_id"]: CityDescription(city) for city, _ in cities_feed}
        self.num_hotels = sum(c.num_hotels for c in self.cities)

    @property
    def cities(self):
        return six.itervalues(self.city_map)


class TaskController(object):
    def __init__(self, num_connections):
        self.yt_mutex = gevent.lock.Semaphore(1)
        self.semaphore = gevent.lock.Semaphore(num_connections)
        self.spawned_tasks = []
        self.num_completed = 0

    def __enter__(self):
        return self

    def __exit__(self, *args):
        gevent.joinall(self.spawned_tasks)
        for task in self.spawned_tasks:
            if task.exception is not None:
                if isinstance(task.exception, requests.HTTPError) and task.exception.response is not None and task.exception.response.text is not None:
                    LOG.info('HTTPError: {}. Response text (first 10000 chars): {}'.format(task.exception, task.exception.response.text[:10000]))
                raise task.exception

    def spawn(self, task, *args, **kwargs):
        kwargs['task_controller'] = self
        self.spawned_tasks.append(gevent.spawn(task, *args, **kwargs))

    def inform_about_completion(self):
        self.num_completed += 1
        LOG.info('Completed {} tasks out of {}'.format(self.num_completed, len(self.spawned_tasks)))


class Booking21LangsProvider(base.DefaultLangsProvider):
    langs_map = {  # assume that first language is the main.
        "ru": ['ru', 'en'],
        "il": ['he', 'en', 'ru'],  # Israel:  Иврит + английский + русский
        'tr': ['tr', 'en', 'ru'],
        'default': ['en']  # ru - if present, else - en.
    }


class Booking21(base.Partner):
    name = "booking21"

    min_records_count_accepted = 1.2e6
    max_records_added_or_removed_count = 50e3
    warn_records_count_change = 20e3

    yt_run_parameters = dict(memory_limit=1073741824,  # 1 gb instead of 500 mb
                             **base.Partner.yt_run_parameters)
    allowed_fields = base.Partner.allowed_fields + [
        '_chain',
        '_ranking',
        '_reviewCount',
        '_reviewRating',
        '_checkinBeginTime',
        '_checkinEndTime',
        '_checkoutBeginTime',
        '_checkoutEndTime',
    ]
    hidden_fields = base.Partner.hidden_fields + [
        'email',  # +, checked # actually, this is now present according to docs - since api 2.3
        'photos',  # +, checked. There are no photos in 21 feed and photos were forbidden (legally) from old feed.
        'rating',  # todo: check if present in feed
        'actualizationTime',  # todo: check if present in feed
    ]
    publish_features = False

    hotels_table_name = "hotels"
    forbidden_country_codes = [
        'xc',
    ]

    langs_provider = Booking21LangsProvider()
    languages = list(set().union(*six.itervalues(langs_provider.langs_map)))
    merge_translations = True

    url_template = "https://distribution-xml.booking.com/2.4/{data_format}/{target}?"

    feed_fields = [
        "hotel_description",  # missing
        "hotel_facilities",  # present
        "hotel_info",  # present
        "hotel_policies",  # missing
        "payment_details",  # missing
        "room_description",  # missing
        "room_facilities",  # missing
        "room_info",  # missing
        "hotel_photos",  # missing
        "room_photos",  # missing
    ]

    def get_url(self, target, language=None, data_format='xml', **kwargs):
        language = self.languages if language is None else language
        url = self.url_template.format(target=target, data_format=data_format) + ''.join('{}={}&'.format(k, v) for k, v in six.iteritems(kwargs))
        url += "language={}".format(language) if isinstance(language, six.string_types) else '&'.join('languages={}'.format(lang) for lang in language)
        return url

    rubric_map = rubrics_mapping.rubric_map
    hotel_type_map = hotel_type_mapping.hotel_type_map

    def __init__(self, session, args):
        super(Booking21, self).__init__(session, args)
        self.auth = (args.user, args.password)
        self.country_map = dict()
        self._countries = args.countries
        self.results_per_batch = args.results_per_batch
        self.cities_per_batch = args.cities_per_batch
        self.download_by_country = args.download_by_country
        self.cities_count_limit = args.cities_count_limit
        self.num_booking_connections = args.num_booking_connections
        self.hotel_type_id_map = dict()
        self.hotel_theme_type_id_map = dict()
        self.cities_by_country = dict()
        self.hotel_facilities_map = dict()

    @property
    def countries(self):
        if self._countries is None:
            self._countries = list(six.iterkeys(self.country_map))
        return self._countries

    def load_facilities(self):
        successful = True
        self.download_and_save(self.get_url("facilityTypes"), name="facilities", auth=self.auth, limit=parsers.NO_LIMIT)
        with self.download_feed(self.get_url("hotelFacilityTypes"), auth=self.auth, limit=parsers.NO_LIMIT) as hf_feed:
            raw = []
            for facility, extra in hf_feed:
                raw.append((facility, extra))
                try:
                    record = dict()
                    # record['facility_type_id'] = str(facility['facility_type_id'])
                    record['id'] = str(facility['hotel_facility_type_id'])
                    record['name'] = facility['name']
                    record['translations'] = Translations(facility)
                    self.hotel_facilities_map[record['id']] = record
                except:
                    successful = False
            self.save_raw_feed(raw, "hotel_facilities")
        if not successful:
            raise Exception("Loading of hotelFacilityTypes failed")

    def load_country_list(self):
        with self.download_feed(self.get_url('countries'), auth=self.auth, limit=parsers.NO_LIMIT) as countries_feed:
            raw = []
            for country, extra in countries_feed:
                country_obj = CountryDescription(country)
                if country_obj.code in self.forbidden_country_codes:
                    continue
                self.country_map[country_obj.code] = country_obj
                raw.append((country, extra))
            self.save_raw_feed(raw, "countries")

    def load_hotel_types(self):
        with self.download_feed(self.get_url('hotelTypes'), auth=self.auth, limit=parsers.NO_LIMIT) as ht_feed:
            raw = []
            for hotel_type, extra in ht_feed:
                self.hotel_type_id_map[hotel_type['hotel_type_id']] = hotel_type['name']
                raw.append((hotel_type, extra))
            self.save_raw_feed(raw, "hotel_types")

    def load_hotel_theme_types(self):
        with self.download_feed(self.get_url('hotelThemeTypes', language=[]), auth=self.auth, limit=parsers.NO_LIMIT) as ht_feed:
            raw = []
            for hotel_theme_type, extra in ht_feed:
                self.hotel_theme_type_id_map[hotel_theme_type['theme_id']] = hotel_theme_type['name']
                raw.append((hotel_theme_type, extra))
            self.save_raw_feed(raw, "hotel_theme_types")

    def task_download_feed_to_hotels_table(self, url, parser_extra_info, task_controller):
        with task_controller.semaphore:
            with self.download_feed(url, downloader=downloaders.PagedStreamingDownloader(),
                                    parser=parsers.MultiStreamXmlParser(extra_info=parser_extra_info),
                                    auth=self.auth) as feed:
                saved_feed = list(feed)
        with task_controller.yt_mutex:  # todo: don't lock, rather save and download with separate client.
            self.save_raw_feed(saved_feed, self.hotels_table_name, append=True)
        task_controller.inform_about_completion()

    def task_download_cities_by_country(self, country, task_controller):
        with task_controller.semaphore:
            with self.download_feed(self.get_url(target="cities", countries=country, language=self.get_langs(country)),
                                    downloader=downloaders.PagedStreamingDownloader(),
                                    parser=parsers.MultiStreamXmlParser(),
                                    auth=self.auth, limit=parsers.NO_LIMIT) as feed:
                saved_feed = list(feed)
        self.country_map[country].set_cities_feed(saved_feed)
        task_controller.inform_about_completion()

    def download_all_feeds_by_country(self):
        with TaskController(self.num_booking_connections) as task_controller:
            for country in self.countries:
                for lang in self.get_langs(country):
                    task_controller.spawn(self.task_download_feed_to_hotels_table,
                                          self.get_url("hotels", country_ids=country, language=lang, extras=self.feed_fields),
                                          dict(lang=lang))

    def download_all_feeds_by_city(self):
        with TaskController(self.num_booking_connections) as task_controller:
            for country in self.countries:
                task_controller.spawn(self.task_download_cities_by_country, country)

        def spawn_batch_download(ids, lang):
            if len(ids) == 0:
                raise Exception("Empty city ids in hotels request")
            task_controller.spawn(self.task_download_feed_to_hotels_table,
                                  self.get_url(target='hotels', language=lang, city_ids=','.join(ids), extras=','.join(self.feed_fields)),
                                  dict(lang=lang))

        with TaskController(self.num_booking_connections) as task_controller:
            # todo: optimize cities groupings
            for country in sorted(self.countries, key=lambda c: self.country_map[c].num_hotels, reverse=True):
                for lang in self.get_langs(country):
                    ids = []
                    num_hotels = 0
                    for city in list(self.country_map[country].cities)[:self.cities_count_limit]:
                        if (num_hotels + city.num_hotels > self.results_per_batch and num_hotels > 0) or len(ids) > self.cities_per_batch:
                            spawn_batch_download(ids, lang)
                            ids = []
                            num_hotels = 0
                        if city.num_hotels > 0:
                            ids.append(city.id)
                            num_hotels += city.num_hotels
                    if num_hotels > 0:
                        spawn_batch_download(ids, lang)

    def download_all_feeds(self):
        self.load_country_list()
        self.load_hotel_types()
        self.load_facilities()
        self.load_hotel_theme_types()
        self.create_feed_table(self.hotels_table_name, ignore_existing=True)
        if self.download_by_country:
            self.download_all_feeds_by_country()
        else:
            self.download_all_feeds_by_city()

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        arg_group = parser.add_argument_group(Booking21.name)
        arg_group.add_argument("--user", '-u', default="yandex1")
        arg_group.add_argument("--password", '-p', required=True)
        arg_group.add_argument("--countries", '-c', default=None, nargs='+')
        arg_group.add_argument("--results_per_batch", '-b', type=int, default=1000)
        arg_group.add_argument("--cities_per_batch", '-cpb', type=int, default=1000)
        arg_group.add_argument("--cities_count_limit", '-cl', type=int, default=None)
        arg_group.add_argument("--download_by_country", '--dbc', action='store_true', default=False)
        arg_group.add_argument("--num_booking_connections", '--nbc', type=int, default=5)

    def map(self, dict_item, info):
        hotel = objects.Hotel()

        original_id = dict_item.get("hotel_id")
        if original_id is None:
            if "your account is not whitelisted to access them" in dict_item.get("message", ''):
                return None
            raise Exception("Booking21: Unknown type of record without hotel_id attribute")
        hotel.original_id = original_id
        hotel._partner = "booking"

        hotel_data = dict_item["hotel_data"]  # crash if missing - is desired behavior.

        lang = info['lang']
        lang_enum = enums.Language.__getattr__(lang.upper())

        country = hotel_data.get("country")
        if country is None:
            warnings.warn(GeoWarning("Country code is missing"))
            return None
        country = country.lower()
        if country == 'an':
            country = 'cw'  # absurd already non-existing country fix.
        hotel.country = country.upper()

        name = hotel_data.get("name")
        if name is not None:
            hotel.name.set(name, lang=lang_enum)
        else:
            warnings.warn(NameWarning("Hotel has no name value for lang {}".format(lang)))

        location = hotel_data['location']
        hotel.lon = location["longitude"]
        hotel.lat = location["latitude"]

        hotel.zip_index = hotel_data["zip"]

        hotel.star = self.parse_stars(hotel_data.get('class'))  # consider "exact_class" and "class_is_estimated"
        hotel._ranking = hotel_data.get("ranking")
        hotel._review_rating = hotel_data.get("review_score")  # out of 10
        hotel._review_count = hotel_data.get("number_of_reviews")

        checkin_checkout_times = hotel_data.get("checkin_checkout_times", {})
        checkin_begin_time = self.format_time(checkin_checkout_times.get("checkin_from"))
        hotel._checkin_begin_time = checkin_begin_time
        if checkin_begin_time is not None:
            hotel.check_in = features_enums.CheckIn.__getattr__("checkin_" + self.get_time_enum(checkin_begin_time))
        hotel._checkin_end_time = self.format_time(checkin_checkout_times.get("checkin_to"))
        hotel._checkout_begin_time = self.format_time(checkin_checkout_times.get("checkout_from"))
        checkout_end_time = self.format_time(checkin_checkout_times.get("checkout_to"))
        hotel._checkout_end_time = checkout_end_time
        if checkout_end_time is not None:
            hotel.check_out = features_enums.CheckOut.__getattr__("checkout_" + self.get_time_enum(checkout_end_time))

        country_translations = self.country_map[country].translations
        country_name = helpers.to_unicode(country_translations.get(lang, country_translations.get("default")))
        city = helpers.to_unicode(hotel_data.get('city'))
        street_address = helpers.to_unicode(hotel_data.get('address'))

        address_elements = [country_name, city, street_address]
        valid_elements = [element for element in address_elements if element is not None and element != '']
        addr_string = ', '.join(valid_elements)
        hotel.address.set(addr_string, lang=lang_enum)
        if city is not None:
            hotel._city.set(city, lang=lang_enum)
        if lang.lower() == 'en':
            try:
                default_city = helpers.to_unicode(self.country_map[country].city_map[hotel_data['city_id']].translations['default'])
                hotel._city.add(default_city)
            except Exception as e:
                if original_id not in ["4670047", "3220402"]:
                    warning_message = "Can't get default translation for city"
                    if self.debug:
                        warning_message += "\ncity: {}, city_id: {}".format(city.encode('utf-8'), hotel_data.get('city_id'))
                        warning_message += "\nerror: {}".format(e)
                    warnings.warn(LocalizationWarning(warning_message))
        hotel_type_id = hotel_data['hotel_type_id']
        hotel_type = self.hotel_type_id_map.get(hotel_type_id)
        if hotel_type_id not in self.hotel_type_id_map:
            if original_id != "765420":  # known case - id 0 for some reason.
                warnings.warn(HotelTypeWarning("Partner didn't provide mapping for hotel type id {}".format(hotel_type_id)))
        elif hotel_type not in self.rubric_map:
            warnings.warn(RubricWarning("Unknown rubric for hotel type '{}'".format(hotel_type)))
        rubric = self.rubric_map.get(hotel_type, enums.HotelRubric.HOTEL)
        hotel.rubric = rubric
        if hotel_type in self.hotel_type_map:
            for hotel_type_val in self.hotel_type_map.get(hotel_type, []):
                hotel.hotel_type.add(hotel_type_val)

        for phone in six.itervalues(hotel_data.get('telephone') or {}):
            if phone is not None:
                hotel.phone.add(phone, type=enums.PhoneType.PHONE)
        for phone in six.itervalues(hotel_data.get('fax') or {}):
            if phone is not None:
                hotel.phone.add(phone, type=enums.PhoneType.FAX)

        self.generate_label(hotel, 'booking')

        url = hotel_data.get("url")
        if url:
            url += "?aid=350687&label={label}".format(label=hotel.label_hash)
            # note: deep_link aid looks different "deep_link_url": "booking://hotel/12928?affiliate_id=350687",
            hotel.url.add(url, type=enums.UrlType.BOOKING)

        # hotel._chain = hotel_data.get("chain_id")
        if hotel_data.get('is_closed') != '0':
            warnings.warn(ClosedHotelWarning("Hotel is closed"))
        # features from amenities
        if lang.lower() == 'en':
            hotel_facilities = helpers.wrap_into_list(hotel_data.get('hotel_facilities', []))
            facilities = []
            for facility in hotel_facilities:
                facility_desc = self.hotel_facilities_map.get(str(facility['hotel_facility_type_id']))
                if facility_desc is None:
                    warnings.warn(HotelTypeWarning("Unknown facility id {} with name {}".format(facility['hotel_facility_type_id'], facility['name'])))
                    continue
                facility_ru = facility_desc['translations'].get('ru', facility['name'])
                if facility_ru is None:
                    warnings.warn(HotelTypeWarning("Unknown facility id {} with empty name".format(facility['hotel_facility_type_id'])))
                    continue
                facility_ru = facility_ru.encode('utf-8')
                facilities.append(facility_ru)
            features_map = features_mapping.create_features_mapping(hotel)
            # add theme ids to facilities list.
            for theme_id in helpers.wrap_into_list(hotel_data.get('theme_ids') or []):
                if theme_id in self.hotel_theme_type_id_map:
                    facilities.append(self.hotel_theme_type_id_map[theme_id])
                else:
                    warnings.warn(HotelTypeWarning("Partner didn't provide mapping for theme id {}".format(theme_id)))
            self.map_features(features_map, facilities, rubric, debug=self.debug)
        number_of_rooms = hotel_data.get('number_of_rooms')
        if number_of_rooms is not None:
            try:
                hotel.room_number = int(number_of_rooms)
            except ValueError:
                warnings.warn(AmenityWarning("Incorrect 'number_of_rooms' value: {}".format(helpers.format_to_text(number_of_rooms))))

        mandatory_deposit = hotel_data.get('creditcard_required')
        if mandatory_deposit is not None:
            hotel.mandatory_deposit = bool(int(mandatory_deposit))  # "0" -> 0 -> False
        # todo: content to be parsed:
        # 4) Follow-up: Some facilities can specify whether they are free or paid. Though that may vary by room type, so not important.
        # 5) email - do not load (not available)

        # <max_persons_in_reservation>0</max_persons_in_reservation>
        # <max_rooms_in_reservation>3</max_rooms_in_reservation>

        return hotel


if __name__ == '__main__':
    Booking21.main()
