import base64
import csv
import logging
import warnings

from library.python import resource

from travel.hotels.feeders.lib import base, downloaders, parsers, helpers
from travel.hotels.feeders.lib.model import objects, enums
from travel.hotels.feeders.lib.model.log_message_types import RubricWarning, HotelTypeWarning
from travel.hotels.feeders.partners.bronevik.lib import rubrics_mapping, hotel_type_mapping, features_mapping

LOG = logging.getLogger(__name__)


class Bronevik(base.Partner):
    name = "bronevik"

    rubric_map = rubrics_mapping.rubric_map
    hotel_type_map = hotel_type_mapping.hotel_type_map

    min_records_count_accepted = 2  # минимальное количество отелей при котором фид считается успешно построенным
    max_records_added_or_removed_count = None  # максимальное удаленное или добавленное количество отелей
    warn_records_count_change = None

    allowed_fields = base.Partner.allowed_fields + [

    ]  # скрытые поля - начинающиеся с '_' - которые нужно разрешить в этом фиде
    hidden_fields = base.Partner.hidden_fields + [

    ]  # общие поля которых нет в этом фиде и которые можно скрыть.

    bronevik_dump_url = 'https://hotels-api.bronevik.com/dumps/download/'

    def __init__(self, session, args):
        super(Bronevik, self).__init__(session, args)
        self.dump_login = args.dump_login
        self.dump_password = args.dump_password

    def download_all_feeds(self):
        url = self.bronevik_dump_url + '?lang=ru'  # чувствителен к регистру ru или en
        auth = (self.dump_login, self.dump_password)
        self.download_and_save(
            url=url,
            name=self.hotels_table_name,
            downloader=downloaders.StreamingDownloader(),
            parser=parsers.JsonParser(),
            auth=auth,
            headers=self.gen_headers()
        )

    def map(self, item, info):
        return BronevikRowMapper().bronevik_map(
            item,
            info,
            self.rubric_map,
            self.hotel_type_map
        )

    def gen_headers(self):
        user_pass = "{}:{}".format(self.dump_login, self.dump_password)
        base64e_user_pass = base64.b64encode(user_pass)
        return {"Authorization": "Basic {}".format(base64e_user_pass)}

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        arg_group = parser.add_argument_group(Bronevik.name)
        arg_group.add_argument("--dump-login", required=True)
        arg_group.add_argument("--dump-password", required=True)


class BronevikRowMapper(object):
    def __init__(self):
        self.amenities_dict = dict()
        self.download_amenities()

    def download_amenities(self):
        amenities_string = resource.find("/amenities.csv").splitlines()
        reader = csv.DictReader(amenities_string, delimiter=';')
        for amenity in reader:
            self.amenities_dict[amenity["id"]] = helpers.format_to_text(amenity["name"])

    def gen_feature_keys(self, item):
        feature_keys = []
        LOG.info(self.amenities_dict)
        for amenity in item['descriptionDetails']['availableAmenities']['availableAmenity']:
            LOG.info(amenity)
            if amenity["included"]:
                feature_keys.append(self.amenities_dict[str(amenity["id"])])
        return feature_keys

    def bronevik_map(self, item, info, rubric_map, hotel_type_map):
        hotel = objects.Hotel()
        LOG.info(item)
        hotel.original_id = str(item['id'])
        hotel.name = helpers.format_to_text(item['name'])
        hotel.address = "{}, {}, {}".format(
            helpers.format_to_text(item['descriptionDetails']['countryName']),
            helpers.format_to_text(item['cityName']),
            helpers.format_to_text(item['address']))
        country_id = item['descriptionDetails']['countryId']
        hotel.country = 'RU' if int(country_id) == 1 else item['descriptionDetails']["countryName"]
        latitude = item['descriptionDetails']['latitude']
        if latitude is None or latitude == "":
            warnings.warn("Latitude is none")
            latitude = 0.0
        hotel.lat = float(latitude)
        longitude = item['descriptionDetails']['longitude']
        if longitude is None or longitude == "":
            warnings.warn("Longitude is none")
            longitude = 0.0
        hotel.lon = float(longitude)
        hotel_type = item['type']
        LOG.info("hotel type")
        LOG.info(hotel_type)
        if hotel_type not in rubric_map:
            warnings.warn(RubricWarning('Unknown rubric for hotel type {}'.format(hotel_type)))
        rubric = rubric_map.get(item['type'], enums.HotelRubric.HOTEL)
        hotel.rubric = rubric
        if hotel_type not in hotel_type_map:
            warnings.warn(HotelTypeWarning('Unknown hotel type for hotel type {}'.format(hotel_type)))
        for hotel_type_item in hotel_type_map.get(item['type'], []):
            hotel.hotel_type.add(hotel_type_item)
        hotel.zip_index = item['descriptionDetails']['zipCode']
        hotel._partner = 'bronevik'
        hotel.star = base.StarParser.parse_stars(item['category'])
        for photo in item['descriptionDetails']['photos']['photo']:
            hotel.photos.add(link=helpers.format_to_text(photo['url']))
        for room in item['rooms']['room']:
            LOG.info(room)
            room_id = room['id']
            name = helpers.format_to_text(room['name'])
            description = helpers.format_to_text(room['description'])
            amenities = room['availableAmenities']['availableAmenity']
            photos = []
            max_allowed_occupancy = room['roomCapacity']
            area_square_meters = room['size']
            for photo in room['photos']['photo']:
                hotel.photos.add(link=photo['url'])
                photos.append({
                    'link': helpers.format_to_text(photo['url']),
                })
            bed_sets = []
            for bed_set in room['availableBedSets']['bedSet']:
                beds = []
                for bed in bed_set['bed']:
                    beds.append({
                        'type': bed['type'],
                        'amount': bed['amount'],
                    })
                bed_sets.append(beds)
            hotel.room_types.add(
                id=room_id,
                lang=enums.Language.RU,
                value=name,
                description=description,
                amenities=amenities,
                max_allowed_occupancy=max_allowed_occupancy,
                area_square_meters=area_square_meters,
                photos=photos,
                bronevik_bed_sets=bed_sets
            )
        features_map = features_mapping.create_features_mapping(hotel)
        facilities = self.gen_feature_keys(item)
        base.FeatureMapper.map_features(features_map, facilities, rubric)

        return hotel
