# coding=utf-8
import json
import logging
import warnings

from requests import HTTPError
from travel.hotels.feeders.lib import base, downloaders, parsers, helpers
# Для всех ворнингов нужно использовать один из следующих типов (можно добавлять)
from travel.hotels.feeders.lib.model import log_message_types
from travel.hotels.feeders.lib.model import objects, enums
from travel.hotels.feeders.partners.dolphin import hotel_type_mapping, rubrics_mapping

TYPES_WITHOUT_EXCURSIONS = "0,12,20,53,56,57,58,64"

LOG = logging.getLogger(__name__)


class Dolphin(base.Partner):
    name = "dolphin"

    min_records_count_accepted = None  # минимальное количество отелей при котором фид считается успешно построенным
    max_records_added_or_removed_count = None  # максимальное удаленное или добавленное количество отелей
    warn_records_count_change = None

    allowed_fields = base.Partner.allowed_fields + [
        "_dolphinRegion", "_dolphinDistrict", "_dolphinCity", "_dolphinOriginalName", "_dolphinOriginalAddress"
    ]  # скрытые поля - начинающиеся с '_' - которые нужно разрешить в этом фиде
    hidden_fields = base.Partner.hidden_fields + []  # общие поля которых нет в этом фиде и которые можно скрыть.

    base_url = "https://www.delfin-tour.ru/jsonyandex/Subagents/"
    hotel_ids = []
    geo = {}

    rubric_map = rubrics_mapping.rubric_map
    hotel_type_map = hotel_type_mapping.hotel_type_map

    def load_areas(self):
        with self.download_feed(url=self.base_url + "Areas",
                                downloader=downloaders.StreamingDownloader(),
                                parser=parsers.JsonParser(),
                                limit=parsers.NO_LIMIT) as areas:
            def wrapped_feed():
                for item, extra in areas:
                    area_type = item["type"]
                    id = item["id"]
                    if item['title']:
                        item['title'] = helpers.to_utf8(item['title'])
                    self.geo.setdefault(area_type, {})[id] = item
                    yield item, extra

            self.save_raw_feed(wrapped_feed(), "areas")

    def load_hotel_ids(self):
        def wrapped_feed():
            for item, extra in feed:
                yield (item, extra)
                self.hotel_ids.append(item)
            LOG.info("Total hotels count: {}".format(len(self.hotel_ids)))

        with self.download_feed(url=self.base_url + "HotelsIdsNew?tour_type=" + TYPES_WITHOUT_EXCURSIONS,
                                downloader=downloaders.StreamingDownloader(),
                                parser=parsers.JsonParser()) as feed:
            self.save_raw_feed(wrapped_feed(), "hotel_ids")

    def load_hotels(self):
        def try_hotel(hotel_id, attempts):
            last_error = None
            for attempts in xrange(attempts):
                try:
                    with helpers.all_logging_disabled():
                        with self.download_feed(url=self.base_url + "HotelContent?id={}".format(hotel_id),
                                                downloader=downloaders.StreamingDownloader(),
                                                parser=parsers.DelimitedStreamParser()) as feed:
                            item, extra = next(feed)
                            return json.loads(item), extra
                except HTTPError as e:
                    if e.response is not None and e.response.status_code == 204:
                        # no sense to retry 204's: there is no content
                        raise
                    else:
                        LOG.warn("Error {} for hotel {}, will retry".format(str(e), hotel_id))
                        last_error = e
                        continue
            if last_error:
                raise last_error

        def hotels_feed():
            failed_downloads = []
            loaded_hotels_count = 0
            for hotel_id in self.hotel_ids:
                try:
                    yield try_hotel(hotel_id, 5)
                    loaded_hotels_count += 1
                    if self.limit is not None and loaded_hotels_count >= self.limit:
                        break
                    if loaded_hotels_count % 100 == 0:
                        LOG.info("Parsed %s JSON items", loaded_hotels_count)
                except HTTPError:
                    failed_downloads.append(hotel_id)
            LOG.info("Parsed %s JSON items; done!", loaded_hotels_count)
            LOG.warn("Failed download of following ids: {}".format(failed_downloads))

        self.save_raw_feed(hotels_feed(), "hotels")

    def download_all_feeds(self):
        self.load_areas()
        self.load_hotel_ids()
        self.load_hotels()

    def map(self, dict_item, info):  # dict item - питоновский dict из колонки item в hotels табличке
        hotel = objects.Hotel()
        hotel.original_id = str(dict_item["Id"])
        category_type = helpers.to_utf8(dict_item["Type"])
        if category_type in self.rubric_map:
            rubric = self.rubric_map[category_type]
            hotel_types = self.hotel_type_map.get(category_type, [])
        else:
            warnings.warn("Unknown type " + category_type, log_message_types.RubricWarning)
            return None
        hotel.rubric = rubric
        for hotel_type in hotel_types:
            hotel.hotel_type.add(hotel_type)

        original_name = dict_item["Name"]
        if not original_name:
            warnings.warn("Empty name", log_message_types.NameWarning)
            return None

        hotel.name.add(self.parse_hotel_name(original_name, dict_item["Type"]), lang=enums.Language.RU)
        country = self.geo.get("Country", {}).get(dict_item["Country"]).get("title", "")
        city = self.geo.get("City", {}).get(dict_item["City"], {}).get("title", "")
        region = self.geo.get("Region", {}).get(dict_item["Region"], {}).get("title", "")
        district = self.geo.get("Distinct", {}).get(dict_item["Distinct"], {}).get("title", "")
        address = dict_item["Adress"]
        if address:
            prefix = ", ".join([comp for comp in [country, region, city, district]
                                if comp and comp not in self.unify_address(address)])
            if prefix:
                address = prefix + ", " + address
        else:
            address = ", ".join([comp for comp in [country, region, city, district] if comp])
        l_images = dict_item.get("Photos1020x700", [])
        m_images = dict_item.get("Photos800x600", [])
        s_images = dict_item.get("Photos200x150", [])
        if l_images or m_images or s_images:
            if not (l_images and m_images and s_images):
                warnings.warn("Not all sizes of images are present", log_message_types.ImagesWarning)
            else:
                if len(l_images) != len(m_images) or len(l_images) != len(s_images) or len(m_images) != len(s_images):
                    warnings.warn("Unexpected count of images", log_message_types.ImagesWarning)
                else:
                    for image in l_images:
                        hotel.photos.add(link=image)
        stars = dict_item["Stars"]
        if stars is not None:
            hotel.star.set(base.StarParser.star_map.get(stars))
        hotel.address.add(address, lang=enums.Language.RU)
        hotel.lat = float(dict_item["Latitude"])
        hotel.lon = float(dict_item["Longitude"])
        hotel.country = self.country_code_by_name.get(helpers.to_unicode(country))

        hotel._dolphin_original_name = original_name
        hotel._dolphin_original_address = dict_item["Adress"]
        hotel._dolphin_region = region
        hotel._dolphin_city = city
        hotel._dolphin_district = district
        return hotel

    def __init__(self, session, args):
        super(Dolphin, self).__init__(session, args)

    @staticmethod
    def unify_address(address):
        return helpers.to_utf8(address.replace("р-н", "район").replace("р-он", "район").replace("обл.", "область"))

    @staticmethod
    def parse_hotel_name(original_name, type):
        second_quote = -1
        first_quote = original_name.find('"')
        if first_quote != -1:
            second_quote = original_name.find('"', first_quote + 1)
        if type and first_quote != -1 and second_quote != -1:
            first_quoted_word = original_name[first_quote + 1:second_quote]
            return '{} "{}"'.format(type, first_quoted_word)
        else:
            return original_name

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        pass


if __name__ == '__main__':
    Dolphin.main()
