import gevent
import warnings
import six
import json
import requests
import datetime
import logging
import time

from library.python import resource

from travel.hotels.feeders.lib import base, downloaders, parsers, helpers
from travel.hotels.feeders.lib.base import ytlib
from travel.hotels.feeders.lib.common.data_helpers import merge_photo_duplicates
from travel.hotels.feeders.lib.model import enums, features_enums, objects
from travel.hotels.feeders.lib.model.log_message_types import DictionaryWarning, RubricWarning
from travel.hotels.feeders.lib.model.log_message_types import DebugWarning, AgeWarning
from travel.hotels.feeders.partners.expedia.lib.helpers import gen_expedia_headers
from travel.hotels.feeders.partners.expedia.generated import features_mapping, hotel_type_mapping, rubrics_mapping

LOG = logging.getLogger(__name__)


def get_expedia_link(api_key, secret, lang="en-US", test=True, customer_ip="127.0.0.1"):
    LOG.info('Getting link for %s', lang)
    url = "https://{base}.ean.com/2.4/files/properties/content".format(base='test' if test else 'api')
    params = {
        'language': lang,
        'payment_terms': 'EAC_SA',
        'partner_point_of_sale': 'Standalone',
    }
    rsp = requests.get(url, params=params, headers=gen_expedia_headers(api_key=api_key, secret=secret, customer_ip=customer_ip))
    rsp.raise_for_status()
    link = rsp.json()["href"]

    return link


class TaskController(object):
    def __init__(self, num_connections):
        self.yt_mutex = gevent.lock.Semaphore(1)
        self.semaphore = gevent.lock.Semaphore(num_connections)
        self.spawned_tasks = []
        self.num_completed = 0

    def __enter__(self):
        return self

    def __exit__(self, *args):
        gevent.joinall(self.spawned_tasks)
        for task in self.spawned_tasks:
            if task.exception is not None:
                raise task.exception

    def spawn(self, task, *args, **kwargs):
        kwargs['task_controller'] = self
        self.spawned_tasks.append(gevent.spawn(task, *args, **kwargs))

    def inform_about_completion(self):
        self.num_completed += 1
        LOG.info('Completed {} tasks out of {}'.format(self.num_completed, len(self.spawned_tasks)))


class ExpediaLangsProvider(base.DefaultLangsProvider):
    langs_map = {  # assume that first language is the main.
        "ru": ['ru', 'en'],
        "il": ['he', 'en', 'ru'],  # Israel:  Иврит + английский + русский
        'tr': ['tr', 'en', 'ru'],
        # French speaking countries https://en.wikipedia.org/wiki/List_of_territorial_entities_where_French_is_an_official_language
        "ci": ['fr', 'en'],  # "Côte d'Ivoire": Французский + Английский = # Ivory Coast
        "cd": ['fr', 'en'],  # DR Congo
        "fr": ['fr', 'en'],  # France
        "ca": ['fr', 'en'],  # Canada
        "mg": ['fr', 'en'],  # Madagascar
        "cm": ['fr', 'en'],  # Cameroon
        "ne": ['fr', 'en'],  # Niger
        "bf": ['fr', 'en'],  # Burkina Faso
        "ml": ['fr', 'en'],  # Mali
        "sn": ['fr', 'en'],  # Senegal
        "td": ['fr', 'en'],  # Chad
        "gn": ['fr', 'en'],  # Guinea
        "rw": ['fr', 'en'],  # Rwanda
        "be": ['fr', 'en'],  # Belgium
        "bi": ['fr', 'en'],  # Burundi
        "bj": ['fr', 'en'],  # Benin
        "ht": ['fr', 'en'],  # Haiti
        "ch": ['fr', 'en'],  # Switzerland
        "tg": ['fr', 'en'],  # Togo
        "cf": ['fr', 'en'],  # Central African Republic
        "cg": ['fr', 'en'],  # Congo
        "ga": ['fr', 'en'],  # Gabon
        "gq": ['fr', 'en'],  # Equatorial Guinea
        "dj": ['fr', 'en'],  # Djibouti
        "km": ['fr', 'en'],  # Comoros
        "lu": ['fr', 'en'],  # Luxembourg
        "vu": ['fr', 'en'],  # Vanuatu
        "sc": ['fr', 'en'],  # Seychelles
        "mc": ['fr', 'en'],  # Monaco
        "fi": ['fi', 'en'],  # Финляндия: Финский + Английский
        "lv": ['lv', 'en', 'ru'],  # Латвия: Латышский + Английский + Русский
        "lt": ['lt', 'en', 'ru'],  # Литва: Литовский + Английский + Русский
        'default': ['en']  # ru - if present, else - en.
    }


class Expedia(base.Partner):
    name = "expedia"

    min_records_count_accepted = 200e3
    max_records_added_or_removed_count = 60e3
    warn_records_count_change = 30e3

    allowed_fields = base.Partner.allowed_fields + [
        '_reviewCount',
        '_reviewRating',
        '_checkinBeginTime',
        '_checkinEndTime',
        '_checkoutEndTime',
        '_ranking'
    ]

    hidden_fields = base.Partner.hidden_fields + [
        'email',  # todo: check. Missing in first 20 records, but who knows.
        'url',  # todo: to check
        # 'rating',  # todo: take from _expediaRating?
        'chainId',  # todo: check in feed?
    ]

    merge_localized_fields = base.Partner.merge_localized_fields + [
        '_expediaDescriptionsRooms',
        '_expediaDescriptionsDining',
        '_expediaDescriptionsLocation',
        '_expediaDescriptionsAttractions',
        '_expediaDescriptionsRenovations',
        '_expediaDescriptionsBusinessAmenities',
        '_expediaDescriptionsAmenities',
    ]

    merge_dict_fields = base.Partner.merge_dict_fields + [
        'photos',
    ]

    expedia_langs_map = {
        "ar": "ar-SA",
        "zh": "zh-CN",
        # "zh": "zh-TW",  #  Chinese (Traditional)
        "hr": "hr-HR",
        "cs": "cs-CZ",
        "da": "da-DK",
        "nl": "nl-NL",
        "en": "en-US",
        "fi": "fi-FI",
        "fr": "fr-FR",
        # "fr": "fr-CA",  #  French (Canada)
        "de": "de-DE",
        "el": "el-GR",
        "hu": "hu-HU",
        "is": "is-IS",
        "id": "id-ID",
        "it": "it-IT",
        "ja": "ja-JP",
        "ko": "ko-KR",
        "lt": "lt-LT",
        "ms": "ms-MY",
        "nb": "nb-NO",
        "pl": "pl-PL",
        # "pt": "pt-BR",  #  Portuguese (Brazil)
        "pt": "pt-PT",
        "ru": "ru-RU",
        "sk": "sk-SK",
        # "es": "es-MX",  #  Spanish (Mexico)
        "es": "es-ES",
        "sv": "sv-SE",
        "th": "th-TH",
        "tr": "tr-TR",
        "uk": "uk-UA",
        "vi": "vi-VN",
    }

    # Values here should be same as in:
    # https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/lib/java/partner_parsers/src/main/java/ru/yandex/travel/hotels/common/partners/expedia/KnownPropertyAmenity.java?rev=6599845#L17
    breakfast_included_amenities = [
        '1073744141',  # "Free self-serve breakfast"
        '1073744462',  # "Free Korean breakfast"
        '1073744464',  # "Free Japanese breakfast"
        '1073744465',  # "Free Taiwanese breakfast"
        '1073744466',  # "Free Cantonese breakfast"
        '2001',  # "Free breakfast"
    ]

    langs_provider = ExpediaLangsProvider()
    languages = [l for l in list(set().union(*six.itervalues(langs_provider.langs_map))) if l in expedia_langs_map]
    merge_translations = True

    @staticmethod
    def parse_min_age(min_age):
        if min_age is None:
            return None
        age_prefix = 'Age '
        if str(min_age).startswith(age_prefix):
            warnings.warn(AgeWarning("Prefix in min age: {}".format(min_age)))
            min_age = min_age[len(age_prefix):]
        return min_age

    @staticmethod
    def parse_time(t, debug=False, lang=None):
        if t is None:
            return None
        if debug:
            warnings.warn(DebugWarning("Expedia time format: {}, lang: {}".format(helpers.format_to_text(t), lang)))
        dt = t
        if isinstance(t, six.string_types):
            if t.isdigit():
                dt = datetime.time(hour=int(t))
            elif helpers.to_unicode(t) in ['midnight', helpers.to_unicode('полночь'), '24:00']:
                dt = datetime.time(hour=int(0))
            elif helpers.to_unicode(t) in ['noon', 'midday', helpers.to_unicode('полдень')]:
                dt = datetime.time(hour=int(12))
        return Expedia.format_time(dt)

    def __init__(self, session, args):
        super(Expedia, self).__init__(session, args)
        self.api_key = args.api_key
        self.secret = args.secret
        self.num_connections = args.num_connections
        self.languages = args.languages

        self.row_mapper = ExpediaRowMapper(self.name, self.debug, int(time.time()))

    def task_download_feed(self, lang, url_get_callable, feed_table_name, parser_extra_info, task_controller):
        with task_controller.semaphore:
            self.download_and_save(url=url_get_callable(lang),
                                   name=feed_table_name,
                                   downloader=downloaders.StreamingDownloader(decompress=True),
                                   parser=parsers.DelimitedStreamParser(stream_chunk_size=256 * 1024, sep='\n', extra_info=parser_extra_info),
                                   separate_client=True
                                   )
        task_controller.inform_about_completion()

    def download_all_feeds(self):
        self.create_feed_table(self.hotels_table_name, ignore_existing=True)
        feed_tables = []
        with TaskController(self.num_connections) as task_controller:
            for lang in self.languages:
                feed_table_name = self.hotels_table_name + '_' + lang
                elang = self.expedia_langs_map[lang]
                task_controller.spawn(
                    self.task_download_feed,
                    elang,
                    url_get_callable=lambda l: get_expedia_link(api_key=self.api_key, secret=self.secret, lang=l, test=False),
                    feed_table_name=feed_table_name,
                    parser_extra_info=dict(lang=lang),
                )
                feed_tables.append(self.get_raw_table_path(feed_table_name))
            # automatically joins all on exit
        # merge tables
        with ytlib.hide_sys_args():
            ytlib.yt.run_merge(feed_tables, self.get_raw_table_path(self.hotels_table_name), mode="unordered", spec={'combine_chunks': True})
            for table in feed_tables:
                ytlib.yt.remove(table)

    def map(self, dict_item, info):
        return self.row_mapper.map(dict_item, info)

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        arg_group = parser.add_argument_group(Expedia.name)
        arg_group.add_argument("--secret", required=True)
        arg_group.add_argument("--api-key", required=True)
        arg_group.add_argument("--num_connections", '--nc', type=int, default=10)
        arg_group.add_argument("--languages", "--lang", default=Expedia.languages, nargs='+')

    @staticmethod
    def get_format(feature):
        return feature.rsplit(' - ', 1)[0] + ' - {value}'

    @staticmethod
    def get_features(features_keys, features_map):
        features = []
        unknown = []
        if type(features_keys) is six.binary_type:
            features_keys = [features_keys]
        for feature in features_keys:
            if feature in features_map:
                features += features_map[feature]
            elif Expedia.get_format(feature) in features_map:
                # amenity = 'Something - {value}'
                for pair in features_map[Expedia.get_format(feature)]:
                    if pair[1] == '{value}':
                        value = feature.split(' - ')[-1] if ' - ' in feature else '1'  # e.g. 'Desk' -> 'Desk - 1'
                        features.append((pair[0], value))
                    else:
                        features.append(pair)
            else:
                unknown.append(feature)
        return features, unknown


class ExpediaRowMapper(object):
    langs_provider = ExpediaLangsProvider()

    def __init__(self, name, debug, timestamp):
        self.name = name
        self.debug = debug
        self.timestamp = timestamp

        self.country_by_code = json.loads(resource.find('/country_mapping.json'))
        self.category_map = json.loads(resource.find('/category_mapping.json'))
        self.hotel_type_map = hotel_type_mapping.hotel_type_map
        self.rubric_map = rubrics_mapping.rubric_map

    def get_langs(self, country):
        return self.langs_provider.get_langs(country)

    def get_country_by_code(self, code):
        if code not in self.country_by_code:
            if code is not None:
                warnings.warn(DictionaryWarning("Country code '{}' is not in country_mapping dictionary".format(code)))
            return code
        return self.country_by_code[code]

    def get_name(self):
        return self.name

    def get_rubric_map(self):
        return self.rubric_map

    def get_category_map(self):
        return self.category_map

    def get_hotel_type_map(self):
        return self.hotel_type_map

    def get_pansion_by_amenities(self, amenity_ids):
        for amenity_id in amenity_ids:
            if str(amenity_id) in Expedia.breakfast_included_amenities:
                return 'PT_BB'
        return None

    def parse_stars(self, s):
        return base.StarParser.parse_stars(s)

    def get_time_enum(self, time_value):
        return base.TimeEnumParser.get_time_enum(time_value)

    def map_features(self, features_map, facilities, rubric, debug=False):
        return base.FeatureMapper.map_features(features_map, facilities, rubric, debug)

    def map(self, dict_item, info):
        dict_item = json.loads(dict_item)

        address = dict_item['address']
        country = address.get('country_code')
        lang = info['lang']

        if lang.lower() not in self.get_langs(country):
            return
        lang_enum = enums.Language.__getattr__(lang.upper())

        hotel = objects.Hotel()

        hotel.country = country
        country_str = self.get_country_by_code(country)

        hotel.name.set(dict_item['name'], lang=lang_enum)
        hotel.original_id = dict_item['property_id']
        hotel._partner = self.get_name()
        category_type = dict_item["category"]["name"].strip()
        if category_type not in self.get_rubric_map():
            category_id = str(dict_item["category"]["id"]).strip()
            if category_id not in self.get_category_map():
                warnings.warn(RubricWarning("Partner didn't provide mapping for category_id '{}'".format(helpers.format_to_text(category_id))))
                return  # unknown rubric
            category_type = self.get_category_map()[category_id]
        rubric = self.get_rubric_map().get(category_type, enums.HotelRubric.HOTEL)
        hotel.rubric = rubric
        if category_type not in self.get_rubric_map():
            warnings.warn(RubricWarning("Unknown rubric for hotel type '{}'".format(helpers.format_to_text(category_type))))
        for hotel_type_val in self.get_hotel_type_map().get(category_type, []):
            hotel.hotel_type.add(hotel_type_val)

        city = address.get('city')
        line_1 = address.get('line_1')
        line_2 = address.get('line_2')
        address_elements = [country_str, city, line_1, line_2]
        valid_elements = [element for element in address_elements if element is not None and element != '']
        hotel.address.set(', '.join(valid_elements), lang=lang_enum)
        hotel._city.set(city, lang=lang_enum)
        hotel.zip_index = address.get('postal_code')
        coordinates = dict_item['location']['coordinates']
        hotel.lat = coordinates['latitude']
        hotel.lon = coordinates['longitude']

        hotel.phone = dict_item.get('phone')
        fax = dict_item.get('fax')
        if fax is not None:
            hotel.phone.add(fax)

        hotel.actualization_time = dict_item.get('dates', {}).get('updated')

        ratings = dict_item.get('ratings')
        if ratings is not None:
            if ratings.get('property', {}).get('type') == "Star":
                #  Returns a value of either “Star” or "Alternate".
                #  Star indicates the rating is provided by the property’s local star rating authority.
                #  Alternate indicates that the rating is an Expedia-assigned value; an official rating was not available.
                hotel.star = self.parse_stars(ratings.get('property', {}).get('rating'))  # according to https://en.wikipedia.org/wiki/Hotel_rating x.5 ~ x+1
                # more info: https://www.expedia.com/Hotel-Star-Rating-Information
            else:
                hotel.rating = ratings.get('property', {}).get('rating')  # ok for now. but maybe this is still equivalent of stars.
            hotel._review_count = ratings.get('tripadvisor', {}).get('count')
            review_rating = ratings.get('tripadvisor', {}).get('rating')
            if review_rating is not None:
                review_rating = float(review_rating) * 2  # rating out of 10.
            hotel._review_rating = review_rating  # why is this always multiple of 0.5? Rounded up?

        for photo_link in self._photo_links_from_subdict(dict_item.get("images", [])):
            hotel.photos.add(link=photo_link)

        if lang.lower() == 'en':  # different localizations have different ranking..
            hotel._ranking = dict_item.get('rank')
        hotel._expedia_onsite_payment_currency = dict_item.get('onsite_payments', {}).get('currency')
        hotel._expedia_airports_preferred_code = dict_item.get('airports', {}).get('preferred', {}).get('code')

        descriptions = dict_item.get('descriptions')
        if descriptions is not None:
            def set_if_present(subj, obj):
                if subj is not None:
                    obj.set(subj, lang=lang_enum)

            set_if_present(descriptions.get("rooms"), hotel._expedia_descriptions_rooms)
            set_if_present(descriptions.get("dining"), hotel._expedia_descriptions_dining)
            set_if_present(descriptions.get("location"), hotel._expedia_descriptions_location)
            set_if_present(descriptions.get("attractions"), hotel._expedia_descriptions_attractions)
            set_if_present(descriptions.get("renovations"), hotel._expedia_descriptions_renovations)
            set_if_present(descriptions.get("business_amenities"), hotel._expedia_descriptions_business_amenities)
            set_if_present(descriptions.get("amenities"), hotel._expedia_descriptions_amenities)

        if lang.lower() in ['ru', 'en']:
            hotel._expedia_checkin_min_age = Expedia.parse_min_age(dict_item.get('checkin', {}).get('min_age'))
            # time is either hour ('16') or str ('9:00 AM', 'midnight'). Currently decided to convert all into str.
            hotel._checkin_end_time = Expedia.parse_time(dict_item.get('checkin', {}).get('end_time'), self.debug, lang)
            checkin_begin_time = Expedia.parse_time(dict_item.get('checkin', {}).get('begin_time'), self.debug, lang)
            hotel._checkin_begin_time = checkin_begin_time
            if checkin_begin_time is not None:
                hotel.check_in = features_enums.CheckIn.__getattr__("checkin_" + self.get_time_enum(checkin_begin_time))
            checkout_end_time = Expedia.parse_time(dict_item.get('checkout', {}).get('time'), self.debug, lang)
            hotel._checkout_end_time = checkout_end_time
            if checkout_end_time is not None:
                hotel.check_out = features_enums.CheckOut.__getattr__("checkout_" + self.get_time_enum(checkout_end_time))

        amenity_ids = [item['id'] for item in dict_item.get('amenities', {}).values() if "id" in item]
        pansion_by_amenities = self.get_pansion_by_amenities(amenity_ids)
        if pansion_by_amenities is not None:
            hotel._expedia_property_pansion = pansion_by_amenities

        if lang.lower() == 'en':  # only map amenities for EN language
            general_amenities = {helpers.to_unicode(item['name']) for item in dict_item.get('attributes', {}).get('general', {}).values() if "name" in item}
            common_amenities = {helpers.to_unicode(item['name']) for item in dict_item.get('amenities', {}).values() if "name" in item}
            rooms_amenities = set()
            for room in dict_item.get('rooms', {}).values():
                # follow-up: sometimes amenities can contradict to each other, e.g.: no wifi, wifi.
                # Value will depend on the order. In this case we want 'True', but not always.
                room_amenities = {helpers.to_unicode(item['name']) for item in room.get('amenities', {}).values() if "name" in item}
                rooms_amenities.update(room_amenities)

                for index, photo_link in enumerate(self._photo_links_from_subdict(room.get('images', []))):
                    hotel.photos.add(link=photo_link)

            amenities = general_amenities.union(common_amenities).union(rooms_amenities)
            features_map = features_mapping.create_features_mapping(hotel)
            self.map_features(features_map, amenities, rubric, debug=self.debug)

        for room in dict_item.get('rooms', {}).values():
            room_id = room.get('id')
            room_name = room.get('name')

            if not room_id or not room_name:
                continue

            description = room.get('descriptions', {}).get('overview')
            photos = []
            for i, photo in enumerate(room.get('images', [])):
                hero_image = photo.get('hero_image')
                category = photo.get('category')
                link = max(photo['links'].items(), key=lambda kv: int(kv[0].strip('px')))[1]['href']
                # Photos are added for the second time here, but we will deduplicate them several lines later
                hotel.photos.add(link=link)

                photos.append({
                    'hero_image': hero_image,
                    'category': category,
                    'link': link,
                })

            amenities = list(room.get('amenities', {}).values())
            bed_groups = list(room.get('bed_groups', {}).values())
            area = room.get('area', {})
            area_square_meters = area.get('square_meters') or (area['square_feet'] / 10.764 if 'square_feet' in area else None)
            views = list(room.get('views', {}).values())
            max_allowed_occupancy = room.get('occupancy', {}).get('max_allowed')

            hotel.room_types.add(
                id=room_id,
                lang=lang_enum,
                value=room_name,
                description=description,
                photos=photos,
                amenities=amenities,
                area_square_meters=area_square_meters,
                max_allowed_occupancy=max_allowed_occupancy,
                expedia_bed_groups=bed_groups,
                expedia_views=views,
            )

        hotel.photos.values = merge_photo_duplicates(hotel.photos.values)

        return hotel

    @staticmethod
    def _photo_links_from_subdict(subdict):
        if not hasattr(subdict, '__iter__'):
            raise ValueError('Photos container is not iterable')

        for photo in subdict:
            resolutions = {int(val.strip('px')): val for val in photo['links'].keys()}
            max_resolution = max(resolutions.keys())
            yield photo['links'][resolutions[max_resolution]]['href']
