from travel.hotels.feeders.lib import base, downloaders, parsers
from travel.hotels.feeders.lib.common import backa_mapper
from travel.hotels.feeders.lib.model import enums


class Hotels101(base.Partner):
    name = "101hotels"

    min_records_count_accepted = 5000
    max_records_added_or_removed_count = 1000
    warn_records_count_change = 300

    allowed_fields = base.Partner.allowed_fields + [
        '_description',
        '_reviewRating',
        '_refBookingId',
        '_minRate',
        '_currency'
    ]

    hidden_fields = base.Partner.hidden_fields + [
        'zipIndex',
        'chainId',
        'rating',
        'phone',  # +, checked
        'email',
    ]
    url = "https://api.101hotels.ru/documents/yandex/hotels_yandextravel.xml.gz"

    # distinguished_countries = ['RU', 'BY']

    def __init__(self, session, args):
        super(Hotels101, self).__init__(session, args)
        self.auth = (args.user, args.password)

    def download_all_feeds(self):
        self.download_and_save(self.url, "hotels", auth=self.auth,
                               downloader=downloaders.StreamingDownloader(decompress=True),
                               parser=parsers.XmlParser())

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        arg_group = parser.add_argument_group(Hotels101.name)
        arg_group.add_argument("--user", default="ilovehotels")
        arg_group.add_argument("--password", required=True)

    def map(self, item, info):
        hotel = backa_mapper.map_backa(item, debug=self.debug)
        if hotel is None:
            return None
        hotel._partner = "hotels101"  # to correspond with cpa platform

        self.generate_label(hotel, self.name)

        original_id = hotel.original_id.values
        for url in hotel.url.values:
            if url['type'] == enums.UrlType.BOOKING.name:
                url['value'] += "?source=yandextravel&utm_medium=meta&utm_hotel={original_id}&label={label}".format(
                    original_id=original_id, label=hotel.label_hash)

        return hotel


if __name__ == '__main__':
    Hotels101.main()
