# coding=utf-8
import json
import warnings

import six
from library.python import resource
from travel.hotels.feeders.lib import base, downloaders, parsers, helpers
from travel.hotels.feeders.lib.model import objects, enums
from travel.hotels.feeders.partners.hotelscombined2 import features_mapping, hotel_type_mapping, rubrics_mapping
from travel.hotels.lib.python import yqllib, ytlib
from yql.api.v1.client import YqlClient

from travel.hotels.feeders.lib.model.log_message_types import GeoWarning, RubricWarning


class HotelsCombined2(base.Partner):
    name = 'hotelscombined2'

    min_records_count_accepted = 2.5e6
    max_records_added_or_removed_count = 100e3
    warn_records_count_change = 30e3

    allowed_fields = base.Partner.allowed_fields + [
        "_ranking",
        "_reviewRating",
        "_chain",
        "_minRate",
        "_currency"
    ]

    hidden_fields = base.Partner.hidden_fields + [
        "actualizationTime",
    ]

    rubric_map = rubrics_mapping.rubric_map
    hotel_type_map = hotel_type_mapping.hotel_type_map

    theme_id_map = json.loads(resource.find('theme_id.json'))
    property_type_map = json.loads(resource.find('property_type.json'))

    url_template = "https://Datafeeds.Hotelscombined.com/Datafeed.ashx?id={user}&type={target}&PrivateDataKey={password}&OutputType={output_type}&LanguageCode={lang}"
    merge_translations = True

    def get_url(self, lang, target):
        return self.url_template.format(user=self.user, password=self.password, output_type=self.output_type, lang=lang.upper(), target=target)

    def get_hotels_url(self, lang):
        return self.get_url(lang=lang.upper(), target="HOTEL")

    def get_facilities_url(self, lang):
        return self.get_url(lang=lang.upper(), target="FACILITY")

    def __init__(self, session, args):
        super(HotelsCombined2, self).__init__(session, args)
        self.user = args.user
        self.password = args.password
        self.output_type = args.output_type
        self.items_per_page = args.items_per_page
        self.languages = args.languages
        self.facitily_id_map = dict()
        self.yql_client = YqlClient(token=args.yql_token, db=args.yt_proxy) if args.yql_token else None
        self.old_feed_path = args.old_feed_path

    def load_facilities(self):
        for lang in self.languages:
            with self.download_feed(url=self.get_facilities_url(lang), downloader=downloaders.StreamingDownloader(), parser=parsers.XmlParser(extra_info={'lang': lang})) as feed:
                if lang.upper() == "EN":
                    def wrapped_feed():
                        for facility, extra in feed:
                            yield (facility, extra)
                            self.facitily_id_map[facility["FacilityID"]] = facility["FacilityName"]

                    self.save_raw_feed(wrapped_feed(), name="facilities_{}".format(lang.lower()))
                else:
                    self.save_raw_feed(feed, name="facilities_{}".format(lang.lower()))

    def download_all_feeds(self):
        self.load_facilities()
        for lang in self.languages:
            self.download_and_save(
                url=self.get_hotels_url(lang),
                name=self.hotels_table_name,
                downloader=downloaders.PagedStreamingDownloader(
                    rows_key="PageSize", offset_key="PageNo", initial_offset=1, offset_in_pages=True, step=self.items_per_page, limit=5000),
                parser=parsers.MultiStreamXmlParser(items_per_page=self.items_per_page, extra_info={'lang': lang}, stop_on_partial_file=False, prefetch=True),
                append=True
            )

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        arg_group = parser.add_argument_group(HotelsCombined2.name)
        arg_group.add_argument("--user", default="166338")
        arg_group.add_argument("--password", required=True)
        arg_group.add_argument("--output_type", default="XML")
        arg_group.add_argument("--items_per_page", default=50000, type=int)
        arg_group.add_argument("--languages", default=HotelsCombined2.languages, nargs='+')
        arg_group.add_argument("--yql_token", default=None)
        arg_group.add_argument("--old_feed_path", default="//home/travel/prod/feeds/hotelscombined/latest/parsed")

    def map(self, dict_item, info):
        dict_item = {k: v for k, v in six.iteritems(dict_item) if v not in [None, 'Null']}
        hotel = objects.Hotel()

        hotel.original_id = dict_item.get("HotelID")
        hotel._partner = "hotelscombined"

        lang = info['lang']
        lang_enum = enums.Language.__getattr__(lang.upper())

        hotel.name.set(dict_item.get("HotelName"), lang=lang_enum)

        property_type_id = dict_item.get("PropertyTypeID")
        if property_type_id not in self.property_type_map:
            warnings.warn("Partner didn't provide mapping for {}".format(property_type_id))
            return
        hotel_type_name = self.property_type_map[property_type_id]
        if hotel_type_name not in self.rubric_map:
            warnings.warn(RubricWarning("Unknown rubric for hotel type '{}'".format(hotel_type_name)))
        hotel.rubric = rubric = self.rubric_map.get(hotel_type_name, enums.HotelRubric.HOTEL)
        for hotel_type_val in self.hotel_type_map.get(hotel_type_name, []):
            hotel.hotel_type.add(hotel_type_val)

        # address
        hotel.country = dict_item.get("CountryCode", "")
        country = helpers.to_unicode(dict_item.get("CountryName", ""))
        state_name = helpers.to_unicode(dict_item.get("StateName", ""))
        city = helpers.to_unicode(dict_item.get("PlaceName", ""))
        hotel._city.add(city, lang=lang_enum)
        street_address = helpers.to_unicode(dict_item.get("HotelAddress", ""))
        parts = [itm.strip() for itm in street_address.split(',')]  # split street address by ','
        parts = [part.title() if len(part) > 3 else part for part in parts if part.lower() not in [country.lower(), city.lower()]]  # drop country and city
        aggregated_street_address = ''.join(parts)
        if len(aggregated_street_address) == 0:
            warnings.warn(GeoWarning("Hotel has no street address for lang {lang}".format(lang=lang)))
        parts = [country, state_name, city, ] + parts
        address = ", ".join(itm for itm in parts if itm != '')
        hotel.address.set(address, lang=lang_enum)

        hotel.lat = dict_item.get('Latitude')
        hotel.lon = dict_item.get('Longitude')
        hotel.zip_index = dict_item.get("HotelPostcode")

        hotel._chain = helpers.to_unicode(dict_item.get("ChainName"))  # there's also ChainId # todo: this outputs broken values...
        hotel._review_rating = dict_item.get("OverallRating")
        if dict_item.get("SelfRated") == "True":
            hotel.rating = dict_item.get("StarRating")
        else:
            hotel.star = self.parse_stars(dict_item.get("StarRating"))
        hotel._ranking = dict_item.get("Popularity")

        if lang.lower() == 'en' and "Facilities" in dict_item:
            facilities_ids = dict_item["Facilities"].split("|")
            facilities = [self.facitily_id_map[facility_id] for facility_id in facilities_ids]
            # theme_ids
            facilities += [self.theme_id_map.get(theme_id, theme_id) for theme_id in dict_item.get("Themes", "").split('|') if theme_id != ""]
            features_map = features_mapping.create_features_mapping(hotel)
            self.map_features(features_map, facilities, rubric, debug=self.debug)

        self.generate_label(hotel, 'hotelscombined')

        hotel_file_name = dict_item.get('HotelFileName')
        if hotel_file_name is not None:
            hotel.url.add("https://www.hotelscombined.ru/Hotel/{hotel_name}.htm??a_aid=166338&label={label}".format(hotel_name=hotel_file_name, label=hotel.label_hash),
                          type=enums.UrlType.BOOKING)
        # todo: maybe generate direct links to partners. https://www.agoda.com/ru-ru/osobnyak-na-teatralnoy/hotel/kazan-ru.html
        # issue: no label/partner attribution in this case?

        images_by_providers = dict_item.get('ImageSprite')  # "548229605,235,313,OLO|548229606,334,500,OLO",
        if images_by_providers is not None:
            gallery_url = {"gallery_url": "https://www.hotelscombined.ru/Hotel/{hotel_name}.htm??a_aid=166338".format(hotel_name=hotel_file_name)}
            for image_cred in images_by_providers.split('|'):
                ic = image_cred.split(',')
                hotel.photos.add(link="https://edge.media.datahc.com/HI{image_id}.jpg?source={image_source}".format(
                    image_id=ic[0], image_source=ic[-1]
                ), **gallery_url)

        hotel._min_rate = dict_item.get("MinRate")
        hotel._currency = dict_item.get("CurrencyCode")

        return hotel

        # unmapped:
        #  Trademarked: "False"

    def map_raw_table(self, table_name, show_warnings=True):
        super(HotelsCombined2, self).map_raw_table(table_name, show_warnings=show_warnings)
        if self.yql_client is not None:
            parameters = {
                '$output_tables': self.output_hotel_table_paths,
                '$old_feed': self.old_feed_path,
            }
            request = yqllib.run_query(resource.find('join_old_data.yql'), parameters=parameters,
                                       client=self.yql_client, syntax_version=1)
            yqllib.wait_results(request)
            for table_path in self.output_hotel_table_paths:
                ytlib.yt.move(table_path + '_temp', table_path, force=True)

    def prepare(self):
        super(HotelsCombined2, self).prepare()
        if self.yql_client is not None:
            for table_path in self.output_hotel_table_paths:
                ytlib.yt.create("table", table_path + '_temp', attributes={"schema": objects.Hotel.get_yt_schema(partner_name=self.name,
                                                                                                                 publish_features=self.publish_features,
                                                                                                                 hidden_fields=self.hidden_fields,
                                                                                                                 allowed_fields=self.allowed_fields)})


if __name__ == '__main__':
    HotelsCombined2.main()
