# coding=utf-8
import warnings

import six
from travel.hotels.feeders.lib import base, downloaders, parsers, helpers
from travel.hotels.feeders.lib.model import enums, features_enums, objects
from travel.hotels.feeders.partners.ostrovok import features_mapping, hotel_type_mapping, rubrics_mapping

from travel.hotels.feeders.lib.model.log_message_types import RubricWarning, CountryWarning, NameWarning


class OstrovokLangsProvider(base.DefaultLangsProvider):
    langs_map = {  # assume that first language is the main.
        "ru": ['ru', 'en'],
        # german-speaking countries https://en.wikipedia.org/wiki/List_of_territorial_entities_where_German_is_an_official_language
        'de': ['de', 'en', 'ru'],  # Germany
        'be': ['de', 'en', 'ru'],  # Belgium
        'at': ['de', 'en', 'ru'],  # Austria
        'lu': ['de', 'en', 'ru'],  # Luxembourg
        'li': ['de', 'en', 'ru'],  # Liechtenstein
        # spanish-speaking countries https://en.wikipedia.org/wiki/List_of_countries_where_Spanish_is_an_official_language
        'mx': ['es', 'en', 'ru'],  # Mexico
        'co': ['es', 'en', 'ru'],  # Colombia
        'es': ['es', 'en', 'ru'],  # Spain
        'ar': ['es', 'en', 'ru'],  # Argentina
        'pe': ['es', 'en', 'ru'],  # Peru
        've': ['es', 'en', 'ru'],  # Venezuela
        'cl': ['es', 'en', 'ru'],  # Chile
        'ec': ['es', 'en', 'ru'],  # Ecuador
        'gt': ['es', 'en', 'ru'],  # Guatemala
        'cu': ['es', 'en', 'ru'],  # Cuba
        'bo': ['es', 'en', 'ru'],  # Bolivia
        'do': ['es', 'en', 'ru'],  # Dominican Republic
        'hn': ['es', 'en', 'ru'],  # Honduras
        'py': ['es', 'en', 'ru'],  # Paraguay
        'sv': ['es', 'en', 'ru'],  # El Salvador
        'ni': ['es', 'en', 'ru'],  # Nicaragua
        'cr': ['es', 'en', 'ru'],  # Costa Rica
        'pr': ['es', 'en', 'ru'],  # Puerto Rico
        'pa': ['es', 'en', 'ru'],  # Panama
        'uy': ['es', 'en', 'ru'],  # Uruguay
        'gq': ['es', 'en', 'ru'],  # Equatorial Guinea
        # italian-speaking countries https://en.wikipedia.org/wiki/Category:Italian-speaking_countries_and_territories
        'it': ['it', 'en', 'ru'],  # Italy
        'sm': ['it', 'en', 'ru'],  # San Marino
        'hr': ['it', 'en', 'ru'],  # Croatia
        'va': ['it', 'en', 'ru'],  # Vatican
        # it, de
        'ch': ['de', 'it', 'en', 'ru'],  # Switzerland
        'default': ['en', 'ru']  # we have only mapped amenities for ru language.
    }


class Ostrovok(base.Partner):
    name = "ostrovok"

    min_records_count_accepted = 500e3
    max_records_added_or_removed_count = 150e3
    warn_records_count_change = 40e3

    yt_run_parameters = dict(memory_limit=1073741824,  # 1 gb instead of 500 mb
                             **base.Partner.yt_run_parameters)

    allowed_fields = base.Partner.allowed_fields + [
        '_refBookingId',
        '_refExpediaId',
        '_reviewCount',
        '_reviewRating',
        '_checkinBeginTime',
        '_checkoutEndTime',
        '_description',
    ]

    hidden_fields = base.Partner.hidden_fields + [
        'chainId',  # currently not parsed, so far not seen in feed
        'rating',
    ]

    url_by_lang = "https://partner-feedora.s3.eu-central-1.amazonaws.com/common/feed_{lang}.xml.gz"  # langs: it, en, ru, de, es

    rubric_map = rubrics_mapping.rubric_map
    hotel_type_map = hotel_type_mapping.hotel_type_map

    langs_provider = OstrovokLangsProvider()
    allowed_languages = ["en", "ru", "de", "es", "it", ]
    languages = [l for l in list(set().union(*six.itervalues(langs_provider.langs_map))) if l in allowed_languages]
    merge_dict_fields = base.Partner.merge_dict_fields + ["photos", ]
    merge_localized_fields = base.Partner.merge_localized_fields + ['_ostrovokPolicies', ]
    merge_translations = True

    def __init__(self, session, args):
        super(Ostrovok, self).__init__(session, args)

    def download_all_feeds(self):
        for lang in self.languages:  # no-parallel
            self.download_and_save(
                url=self.url_by_lang.format(lang=lang),
                name=self.hotels_table_name,
                downloader=downloaders.StreamingDownloader(decompress=True),
                parser=parsers.XmlParser(extra_info=dict(lang=lang), recover=lang not in ['ru', 'en']),  # try recovering non-essential feeds.
                append=True
            )

    def map(self, dict_item, info):
        hotel = objects.Hotel()
        if isinstance(dict_item, tuple):
            dict_item = dict_item[0]

        hotel.country = country = dict_item['country'].get('@code')
        if country is None:
            warnings.warn(CountryWarning("No country id"))
            return
        lang = info['lang']
        if lang.lower() not in self.get_langs(country):
            return

        original_id = dict_item['@code']
        hotel.original_id = original_id
        hotel._partner = self.name

        lang_enum = enums.Language.__getattr__(lang.upper())
        name = dict_item.get('name')
        if name is not None:
            hotel.name.set(name, lang=lang_enum)
        else:
            warnings.warn(NameWarning("Missing name"))

        hotel.address.set(Ostrovok.get_address(dict_item), lang=lang_enum)
        city = dict_item['city'].get('#text')
        if city is not None:
            hotel._city.set(city, lang=lang_enum)

        category_type = dict_item['type']
        rubric = self.rubric_map.get(category_type, enums.HotelRubric.HOTEL)
        hotel.rubric = rubric
        if category_type not in self.rubric_map:
            warnings.warn(RubricWarning("Unknown rubric for hotel type '{}'".format(category_type)))
        for hotel_type_val in self.hotel_type_map.get(category_type, []):
            hotel.hotel_type.add(hotel_type_val)

        # features from amenities
        # won't do: check if there are amenities present for localizations but not for RU language.
        if lang.lower() == 'ru':  # only map amenities for RU language
            property_ = dict_item.get('amenities', {}).get('property')
            amenities = {} if property_ is None else property_['amenity']
            features_map = features_mapping.create_features_mapping(hotel)
            self.map_features(features_map, amenities, rubric, debug=self.debug)

        hotel.lat = float(dict_item['location']['@lat'])
        hotel.lon = float(dict_item['location']['@lon'])
        if 'numberOfRooms' in dict_item:
            hotel.number_of_rooms = int(dict_item['numberOfRooms'])
        hotel.star = self.parse_stars(dict_item['starRating']['#text'])  # в 80% случаев совпадает со звездами https://yql.yandex-team.ru/Operations/XG6Rc2UOb6G_WXMCeyXxOnXQ6WRuISpEtr1yC2hL_jo=
        # Рейтинг отеля по "пятибальной" шкале (количество «звёзд») - оценка отелей в "звездах". 0 означает отсутствие рейтинга ("звезд")
        # При наличие рейтинга, передается как 10 для 1-ой звезды, 20 для 2-ух звезд, 30 для 3-х звезд, 40 для 4-х звезд, 50 для 5-ти звезд. Если значение не оканчивается на 0 (например, равно 35),
        # то следует его разделить на 10 и математически округлить до целых. Пример: 5 = 1, 4 = 0 , 35 = 4 , 30 = 3 , 32 = 3 , 36 = 4.

        self.generate_label(hotel, self.name)

        url = dict_item['uri']
        url += "?partner_slug=yandex.affiliate.7877&utm_medium=cpa-metasearch&utm_source=yandex_metasearch&utm_term={label}".format(label=hotel.label_hash)
        hotel.url.add(url, type=enums.UrlType.BOOKING)

        hotel.zip_index = dict_item['postcode']
        hotel.phone = dict_item.get('phone')
        email = dict_item.get('email')
        if email is not None and email != "None":
            hotel.email = email  # otherwise "None" string for some reason

        # photos
        for photo in helpers.wrap_into_list((dict_item.get("images") or {}).get("image", [])) + helpers.wrap_into_list((dict_item.get("roomImages") or {}).get("image", [])):
            hotel.photos.add(link=photo["#text"])

        additional = dict_item.get('additionalFields')
        additional = [] if additional is None else additional.get('field', [])
        if isinstance(additional, dict):
            additional = [additional]
        for item in additional:
            name = item['@name']
            text = item['#text']
            if name == 'bookingID':
                hotel._ref_booking_id = text
            if name == 'expediaID':
                hotel._ref_expedia_id = text

        # 1) done. dict_item.get('numberOfRooms')
        number_of_rooms = dict_item.get('numberOfRooms')
        hotel.number_of_rooms = None if number_of_rooms is None else int(number_of_rooms)
        # todo: cool info: 2) dict_item.get('descriptions'),
        # todo: policies
        # 3) done. "yearOpened": "1992", "yearRefurbished": "2014"
        reconstruction_year = dict_item.get('yearRefurbished')
        hotel.reconstruction_year = None if reconstruction_year is None else int(reconstruction_year)
        year_of_foundation = dict_item.get('yearOpened')
        hotel.year_of_foundation = None if year_of_foundation is None else int(year_of_foundation)

        review_count = int(dict_item.get("rating", {}).get("@count", 0))
        if review_count > 0:
            hotel._review_count = review_count
            hotel._review_rating = dict_item.get("rating", {}).get("#text")
        checkin_begin_time = self.format_time(dict_item.get("check_in_time"))  # This data differs across localizations.
        hotel._checkin_begin_time = checkin_begin_time
        if checkin_begin_time is not None:
            hotel.check_in = features_enums.CheckIn.__getattr__("checkin_" + self.get_time_enum(checkin_begin_time))
        checkout_end_time = self.format_time(dict_item.get("check_out_time"))  # currenrt solution:
        hotel._checkout_end_time = checkout_end_time
        if checkout_end_time is not None:
            hotel.check_out = features_enums.CheckOut.__getattr__("checkout_" + self.get_time_enum(checkout_end_time))

        description = dict_item.get("descriptions")
        if description is not None:
            hotel._description.set(description, lang=lang_enum)

        policies = helpers.wrap_into_list((dict_item.get("policies") or {}).get("policy") or [])
        for policy in policies:
            policy_text = policy.get("#text")
            if policy_text is not None:
                hotel._ostrovok_policies.set(policy_text, lang=lang_enum)

        return hotel

    @classmethod
    def get_address(cls, dict_item):
        address = helpers.to_unicode(dict_item['address'])
        city = helpers.to_unicode(dict_item['city'].get('#text'))
        country = helpers.to_unicode(dict_item['country'].get('#text') or cls.country_by_code[dict_item['country']['@code'].upper()][0])
        parts = [itm.strip() for itm in address.split(',')]
        if parts[-1] == city:
            parts = parts[:-1]
        address = ', '.join(parts)
        address = (country, city, address)
        address = (itm.strip() for itm in address if itm is not None and itm.strip() != '')
        return ', '.join(address)


if __name__ == '__main__':
    Ostrovok.main()
