# coding=utf-8
import json
import logging
import warnings
import time

from requests import HTTPError
from library.python import resource
from travel.hotels.feeders.lib import base, downloaders, parsers, helpers
# Для всех ворнингов нужно использовать один из следующих типов (можно добавлять)
from travel.hotels.feeders.lib.common.data_helpers import merge_photo_duplicates
from travel.hotels.feeders.lib.model import objects, enums
from travel.hotels.feeders.lib.model.log_message_types import CoordinatesWarning, HotelIdWarning

LOG = logging.getLogger(__name__)


class Travelline(base.Partner):
    name = "travelline"

    min_records_count_accepted = None  # минимальное количество отелей при котором фид считается успешно построенным
    max_records_added_or_removed_count = None  # максимальное удаленное или добавленное количество отелей
    warn_records_count_change = None

    allowed_fields = base.Partner.allowed_fields + [
        '_travellineEnabledForYandex',
        '_travellineMaxLOS',
    ]  # скрытые поля - начинающиеся с '_' - которые нужно разрешить в этом фиде
    hidden_fields = base.Partner.hidden_fields + [
        'chainId',
        'rating',
    ]  # общие поля которых нет в этом фиде и которые можно скрыть.

    yandex_url = "https://yandex.tlintegration.com/booking/"
    other_url = "https://ibe.tlintegration.com/ApiWebDistribution/BookingForm/"

    yandex_hotel_ids = []
    other_hotel_ids = []

    geo = {}
    languages = [
        "ru-ru",
        "en-us"
    ]
    last_load = None

    merge_translations = True

    def load_yandex_hotel_ids(self):
        def wrapped_feed():
            for hotel_code, extra in feed:
                if hotel_code != "11983":  # skip test hotel
                    yield (hotel_code, extra)
                    self.yandex_hotel_ids.append(hotel_code)
            LOG.info("Total yandex hotels count: {}".format(len(self.yandex_hotel_ids)))

        with self.download_feed(url=self.yandex_url + "hotels",
                                headers={"X-ApiKey": self.api_key},
                                downloader=downloaders.StreamingDownloader(),
                                parser=parsers.JsonParser(ijson_path="hotels.item.code")) as feed:
            self.save_raw_feed(wrapped_feed(), "yandex_hotel_ids")

    def load_other_hotel_ids(self):
        hotel_ids = [hotel_id for hotel_id in resource.find("/hotel_ids.csv").split('\n') if hotel_id]
        yandex_hotel_ids = set(self.yandex_hotel_ids)
        self.other_hotel_ids = [hotel_id for hotel_id in hotel_ids if hotel_id not in yandex_hotel_ids]
        if self.limit is not None:
            limit = self.limit - len(yandex_hotel_ids)
            if limit > 0:
                self.other_hotel_ids = self.other_hotel_ids[:limit]
        LOG.info("Total other hotels count: {}".format(len(self.other_hotel_ids)))
        self.save_raw_feed([(hotel_id, {}) for hotel_id in self.other_hotel_ids], "other_hotel_ids")

    @staticmethod
    def get_yandex_hotel_query_params(hotel_id, language):
        return "hotel_info?hotel.code={}&language={}".format(hotel_id, language)

    @staticmethod
    def get_other_hotel_query_params(hotel_id, language):
        return "hotel_info?hotels[0].code={}&language={}".format(hotel_id, language)

    def on_hotel_loaded(self, hotel, count):
        if count % 100 == 0:
            LOG.info("Parsed %s JSON items", count)
        time_now = time.time()
        if self.last_load is not None:
            time_to_sleep = 0.25 - (time_now - self.last_load)
            if time_to_sleep > 0:
                time.sleep(time_to_sleep)
        self.last_load = time_now

    def load_hotels(self):
        def try_hotel(hotel_id, language, attempts, is_yandex_hotel):
            base_url = self.yandex_url if is_yandex_hotel else self.other_url
            headers = {"X-ApiKey": self.api_key} if is_yandex_hotel else {}
            if is_yandex_hotel:
                params = self.get_yandex_hotel_query_params(hotel_id, language)
            else:
                params = self.get_other_hotel_query_params(hotel_id, language)

            last_error = None
            for attempts in xrange(attempts):
                try:
                    with helpers.all_logging_disabled():
                        with self.download_feed(url=base_url + params,
                                                headers=headers,
                                                downloader=downloaders.StreamingDownloader(),
                                                parser=parsers.DelimitedStreamParser(
                                                    extra_info=dict(lang=language, is_yandex_hotel=is_yandex_hotel, hotel_id=hotel_id))) as feed:
                            item, extra = next(feed)
                            return json.loads(item), extra
                except HTTPError as e:
                    if e.response is not None and e.response.status_code == 204:
                        # no sense to retry 204's: there is no content
                        raise
                    else:
                        LOG.warn("Error {} for hotel {}, will retry".format(str(e), hotel_id))
                        last_error = e
                        continue
            if last_error:
                raise last_error

        def hotels_feed():
            failed_downloads = []
            loaded_hotels_count = 0
            index = 0
            for hotel_id in self.yandex_hotel_ids + self.other_hotel_ids:
                for language in self.languages:
                    try:
                        hotel = try_hotel(hotel_id, language, 5, index < len(self.yandex_hotel_ids))
                        yield hotel
                        loaded_hotels_count += 1
                        if self.limit is not None and loaded_hotels_count >= self.limit:
                            break
                        self.on_hotel_loaded(hotel, loaded_hotels_count)
                    except HTTPError:
                        failed_downloads.append(hotel_id)
                index += 1
            LOG.info("Parsed %s JSON items; done!", loaded_hotels_count)
            LOG.warn("Failed download of following ids: {}".format(failed_downloads))

        self.save_raw_feed(hotels_feed(), "hotels")

    def download_all_feeds(self):
        self.load_yandex_hotel_ids()
        self.load_other_hotel_ids()
        self.load_hotels()

    def map(self, dict_item, info):
        return self.row_mapper.map(dict_item, info)

    def __init__(self, session, args):
        super(Travelline, self).__init__(session, args)
        self.languages = args.languages
        self.api_key = args.api_key
        self.row_mapper = TravellineRowMapper(int(time.time()))

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        arg_group = parser.add_argument_group(Travelline.name)
        arg_group.add_argument("--api-key", required=True)
        arg_group.add_argument("--languages", "--lang", default=Travelline.languages, nargs='+')


class TravellineRowMapper(object):
    language_to_enum = {
        "ru-ru": enums.Language.RU,
        "en-us": enums.Language.EN
    }

    travelline_country_to_code = {
        'RUS': 'RU',
        'ABH': 'AB',
        'BGR': 'BG'
    }

    travelline_country_to_localized_name = {
        'RUS': {
            enums.Language.RU: 'Россия',
            enums.Language.EN: 'Russia'
        },
        'ABH': {
            enums.Language.RU: 'Абхазия',
            enums.Language.EN: 'Abkhazia'
        },
        'BGR': {
            enums.Language.RU: 'Болгария',
            enums.Language.EN: 'Bulgaria'
        }
    }

    def __init__(self, timestamp):
        self.timestamp = timestamp

    @staticmethod
    def remap2to1(dict_item):
        if "hotels" not in dict_item:
            return None
        hotel_item = dict_item["hotels"][0]
        hotel_item["images"] = [{"url": image} for image in hotel_item.get("images")]
        return hotel_item

    def map(self, dict_item, info):
        """
        Map dict structure to Hotel object
        :param dict_item: dict structure from partner
        :param info: meta information about request
        :return: Hotel object or None
        """

        if info["is_yandex_hotel"]:
            hotel_item = dict_item.get("hotel")
        else:
            hotel_item = self.remap2to1(dict_item)
        if hotel_item is None:
            warnings.warn(HotelIdWarning("No hotel data in response"))
            return None
        hotel = objects.Hotel()
        hotel._travelline_enabled_for_yandex = 1 if info["is_yandex_hotel"] else 0
        hotel._travelline_max_los = hotel_item.get('los')
        hotel.original_id = str(hotel_item["code"])
        hotel.rubric = enums.HotelRubric.HOTEL  # TODO fix rubric and add hotel_type

        lang = info["lang"]
        lang_enum = self.language_to_enum[lang]

        hotel.name.add(hotel_item["name"], lang=lang_enum)
        addresses_item = hotel_item["contact_info"]["addresses"]
        if not addresses_item:
            raise Exception("No address was found")

        for address_item in hotel_item["contact_info"]["addresses"]:
            country_name = self.travelline_country_to_localized_name.get(address_item["country_code"], {}).get(
                lang_enum)
            if country_name is None:
                country_name = address_item["country_code"]

            address_str = helpers.to_unicode(country_name) + ", " + helpers.to_unicode(address_item["region"])
            if address_item["region"] != address_item["city_name"]:
                address_str = address_str + ", " + helpers.to_unicode(address_item["city_name"])
            county = address_item.get("county")
            if county and county != address_item["city_name"] and county != address_item["region"]:
                address_str = address_str + ", " + helpers.to_unicode(county)

            for line in address_item["address_line"]:
                if line:
                    address_str = address_str + ", " + helpers.to_unicode(line)
            hotel.address.add(address_str, lang=lang_enum)

        for image in hotel_item["images"]:
            hotel.photos.add(link=image["url"])

        stars = hotel_item.get("stars", None)
        if stars:
            hotel.star.set(base.StarParser.star_map.get(stars))

        hotel.zip_index = addresses_item[0]["postal_code"]
        hotel.email = [email["email_address"] for email in hotel_item["contact_info"]["emails"]]
        for phone in hotel_item["contact_info"]["phones"]:
            hotel.phone.add(phone["phone_number"], type=enums.PhoneType.PHONE)
        if "latitude" in addresses_item[0] and "longitude" in addresses_item[0]:
            hotel.lat = float(addresses_item[0]["latitude"])
            hotel.lon = float(addresses_item[0]["longitude"])
        else:
            warnings.warn(CoordinatesWarning("Missing coordinates"))
        hotel.country = self.travelline_country_to_code.get(addresses_item[0]["country_code"])

        for room in hotel_item.get("room_types", []):
            id = room["code"]
            name = room["name"]
            description = room.get("description")
            amenities = room.get("amenities")
            kind = room.get("kind")
            max_allowed_occupancy = {
                'adults': room.get("max_adult_occupancy"),
                'extra_bed': room.get("max_extra_bed_occupancy"),
            }
            area_square_meters = None
            if room.get("size", {}).get("unit") == "square_metre":
                area_square_meters = room["size"]["value"]
            photos = []

            for index, photo_link in enumerate(map(lambda image: image['url'], room['images'])):
                photos.append({
                    'link': photo_link,
                })
                hotel.photos.add(link=photo_link)

            hotel.room_types.add(
                id=id,
                lang=lang_enum,
                value=name,
                description=description,
                amenities=amenities,
                max_allowed_occupancy=max_allowed_occupancy,
                area_square_meters=area_square_meters,
                photos=photos,
                travelline_kind=kind,
            )

        hotel.photos.values = merge_photo_duplicates(hotel.photos.values)

        return hotel
