# coding=utf-8
import logging
import warnings

from travel.hotels.feeders.lib import base, downloaders, helpers, parsers
from travel.hotels.feeders.lib.model import objects, enums, log_message_types
from travel.hotels.feeders.partners.tvil.generated import features_mapping, rubrics_mapping

LOG = logging.getLogger(__name__)


class Tvil(base.Partner):
    name = "tvil"

    min_records_count_accepted = None
    max_records_added_or_removed_count = None
    warn_records_count_change = None

    allowed_fields = base.Partner.allowed_fields + ["_description"]
    hidden_fields = base.Partner.hidden_fields + ["zipIndex", "email", "phone", "url", "actualizationTime", "rating",
                                                  "roomTypes", "_city", "_partner"]

    def download_locations(self):
        def get_name(location):
            values = dict()
            for value_json in location["name"]:
                values[value_json["@language"]] = value_json["#text"]
            return TvilLocalizedValue(values)

        def cities_feed_with_in_memory_copy():
            for country_json, no_extras in feed:
                country = TvilLocation(code=country_json["@code"], name=get_name(country_json))
                country_json_no_cities = country_json.copy()
                country_json_no_cities.pop("cities", None)
                for city_json in country_json["cities"]["city"]:
                    city = TvilLocation(code=city_json["@code"], name=get_name(city_json), parent=country,
                                        lon=city_json["longitude"], lat=city_json["latitude"])
                    self.cities[city.code] = city
                    city_json_copy = city_json.copy()
                    city_json_copy["country"] = country_json_no_cities
                    yield (city_json_copy, None)

        self.cities = dict()
        with self.download_feed(url=self.locations_url,
                                downloader=downloaders.StreamingDownloader(),
                                parser=parsers.XmlParser()) as feed:
            self.save_raw_feed(cities_feed_with_in_memory_copy(), "cities")
        LOG.info("Cities loaded: {}".format(len(self.cities)))

    def download_hotels(self):
        def feed_with_cities():
            for hotel, no_extras in feed:
                city_id = hotel["location"]["city"]
                if city_id not in self.cities:
                    warnings.warn(log_message_types.CityWarning("Unknown city: {}".format(city_id)))
                    continue
                city = self.cities[city_id]
                yield (hotel, dict(
                    full_city_name=city.get_all_full_names(),
                    country_code=city.parent.code
                ))

        with self.download_feed(url=self.hotels_url,
                                downloader=downloaders.StreamingDownloader(),
                                parser=parsers.XmlParser()) as feed:
            self.save_raw_feed(feed_with_cities(), "hotels")

    def download_all_feeds(self):
        self.download_locations()
        self.download_hotels()

    def map(self, dict_item, info):
        return self.row_mapper.map(dict_item, info)

    def __init__(self, session, args):
        super(Tvil, self).__init__(session, args)
        self.row_mapper = TvilRowMapper()
        self.locations_url = args.locations_url
        self.hotels_url = args.hotels_url

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        arg_group = parser.add_argument_group(Tvil.name)
        arg_group.add_argument("--locations_url", default="https://tvil.ru/files/yatravel/locations.xml")
        arg_group.add_argument("--hotels_url", default="https://tvil.ru/files/yatravel/hotels.xml")


class TvilRowMapper(object):

    def __init__(self):
        pass

    @staticmethod
    def get_lang_enum(lang):
        return enums.Language.__getattr__(lang.upper())

    @staticmethod
    def get_rubric(property_type):
        if property_type in rubrics_mapping.rubric_map:
            return rubrics_mapping.rubric_map[property_type]
        else:
            warnings.warn(log_message_types.RubricWarning("Unknown property type '{}'".format(property_type)))
            return None

    @staticmethod
    def set_features(hotel, dict_item, rubric):
        amenities = dict_item["amenities"]["amenity"] if dict_item["amenities"] is not None else []
        amenity_names = [helpers.to_utf8(amenity["#text"]) for amenity in helpers.wrap_into_list(amenities)]
        features_map = features_mapping.create_features_mapping(hotel)
        base.FeatureMapper.map_features(features_map, amenity_names, rubric)

    def map(self, dict_item, extra_info):
        hotel = objects.Hotel()
        hotel.original_id = dict_item["@code"]
        for name in dict_item["name"]:
            text = name.get("#text")
            language = name.get("@language")
            if text is not None and language is not None:
                language = self.get_lang_enum(language)
                hotel.name.add(text, lang=language)
        for desc in dict_item["descriptions"]["description"]:
            text = desc.get("#text")
            language = desc.get("@language")
            if text is not None and language is not None:
                language = self.get_lang_enum(language)
                hotel._description.add(text, lang=language)
        for address in dict_item["location"]["address"]:
            hotel.address.add(extra_info["full_city_name"][address["@language"]] + ", " + address["#text"],
                              lang=self.get_lang_enum(address["@language"]))
        hotel.lat = float(dict_item["location"]["latitude"])
        hotel.lon = float(dict_item["location"]["longitude"])
        hotel.country = extra_info["country_code"]
        rubric_enum = self.get_rubric(dict_item["property_type"])
        hotel.rubric = rubric_enum
        photos = dict_item["photos"]["photo"] if dict_item["photos"] is not None else []
        if isinstance(photos, dict):
            photos = [photos]
        for photo in photos:
            hotel.photos.add(link=photo["#text"])
        self.set_features(hotel, dict_item, rubric_enum)
        return hotel


class TvilLocation:

    def __init__(self, code, name, parent=None, lon=None, lat=None):
        self.code = code
        self.name = name
        self.parent = parent
        self.lon = lon
        self.lat = lat

    def get_full_name(self, lang):
        name = self.name.get(lang)
        if self.parent is not None:
            name = self.parent.name.get(lang) + ", " + name
        return name

    def get_all_full_names(self):
        all_names = dict()
        for lang in self.name.values.keys():
            all_names[lang] = self.get_full_name(lang)
        return all_names


class TvilLocalizedValue(object):

    def __init__(self, values):
        self.values = values

    def get(self, lang):
        if lang in self.values:
            return self.values[lang]
        else:
            raise Exception("No value for the requested {} language".format(lang))
