# -*- coding: utf8 -*-

import argparse
import logging
from os.path import join

import pandas as pd

from travel.hotels.feeders.lib.common.altay_db import AltayDb

UTF_HEADER = '# -*- coding: utf8 -*-\n'

CODEGEN_WARNING = """
# WARNING, THIS CODE WAS AUTOGENERATED
# DO NOT MODIFY
# TO RE-GENERATE USE $A/travel/hotels/feeders/scripts/amenities_mapping_codegen/

"""

RUBRICS_MAP = {
    30785: 'HOTEL',
    30655: 'DORMITORY',
    31632: 'HOSTEL',
    30788: 'CAMPING_AREA',
    3501492236: 'APARTMENTS_FOR_DAILY_RENT',
    31309: 'RURAL_TOURISM',
    30781: 'RESORT',
    30779: 'REST_HOUSE',
    30791: 'TOURIST_RESORT',
    30787: 'CHILDREN_CAMP',
    3501708107: 'ACCOMMODATION_FOR_DAILY_RENT',
}

MAX_LINE_LEN = 200

INDENTATION = 4


def indent(n):
    return ' ' * n * INDENTATION


class GeneratedCode(object):
    def __init__(self, path):
        self.path = path

    def __enter__(self):
        self.f = open(self.path, 'w')
        self.f.write(UTF_HEADER)
        self.f.write(CODEGEN_WARNING)
        return self.f

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.f.__exit__(exc_type, exc_val, exc_tb)


def generate_available_features(path, altay_db):
    with GeneratedCode(path) as f:
        f.write('from travel.hotels.feeders.lib.model import enums\n\n')
        f.write('# todo: check this at the field value assignment level.\n')
        f.write('available_features = {\n')
        for rubric in sorted(altay_db.rubric_tree, key=lambda k: RUBRICS_MAP[k]):
            f.write('    enums.HotelRubric.{}: [  # {}\n'.format(RUBRICS_MAP[rubric], rubric))
            for permalink in sorted(altay_db.rubric_tree[rubric].keys()):
                f.write('        "{}",\n'.format(permalink))
            f.write('    ],\n')
        f.write('}\n')


def generate_features_enums(path, altay_db):
    with GeneratedCode(path) as f:
        f.write('from enum import Enum\n\n\n')
        for permalink, feature in sorted(altay_db.extracted_features.items()):
            if feature['value_type'] != "dict_value":
                continue
            f.write("""class {class_name}(Enum):""".format(class_name=AltayDb.format_class_name(permalink)))
            f.write("  # Feature https://altay.yandex-team.ru/features/{feature_id}\n".format(feature_id=feature['id']))

            for feature_value_id, feature_value in sorted(feature['values'].items(), key=lambda item: AltayDb.format_enum_name(item[1])):
                value_name = AltayDb.format_enum_name(feature_value)
                f.write('    {} = "{}"  # {}\n'.format(value_name, feature_value, feature_value_id))
            f.write('\n\n')


def format_dict(d):
    return ', '.join("{}={}".format(k, v) for k, v in d.items())


field_type_map = {
    'logical_value': 'BooleanFeatureField',
    'dict_value': 'EnumValueFeatureField',
    'number': 'FeatureField',
    'number_range': 'MinMaxFeatureField',
    'special': 'Field',
}


def generate_hotel_features(path, altay_db):
    with GeneratedCode(path) as f:
        f.write('from travel.hotels.feeders.lib.model import fields, features_enums\n\n\nclass HotelFeatures(object):')
        value_types = set()
        for value_type, permalink, feature in sorted([(v['value_type'], k, v) for k, v in altay_db.extracted_features.items()]):
            name = AltayDb.format_feature_name(permalink)
            field_type = field_type_map.get(value_type)
            kwargs = {'name': '"{}"'.format(permalink)}
            if feature.get('is_multiple'):
                kwargs['allow_multi'] = True

            if value_type == "dict_value":
                kwargs['enum_type'] = 'features_enums.{}'.format(AltayDb.format_class_name(permalink))
            elif value_type == "number":
                kwargs['type_class'] = 'int'
            elif value_type == "number_range":
                logging.info("Skipping number range feature. Value type: {}, permalink: {}".format(value_type, permalink))
                # kwargs['type_class'] = 'int' # or 'float'?
                continue
            elif value_type not in ['logical_value']:  # 'special',
                logging.warn("Not parsed feature. Value type: {}, permalink: {}".format(value_type, permalink))
                continue
            # link to feature in altay
            if value_type not in value_types:
                value_types.add(value_type)
                f.write('\n    # {} features\n'.format(value_type))
            f.write("    # Feature {} https://altay.yandex-team.ru/features/{feature_id}\n".format(name, feature_id=feature['id']))
            f.write('    {name} = fields.{field_type}({kwargs})\n'.format(name=name, field_type=field_type, kwargs=format_dict(kwargs)))


boolean_feature_value_map = {
    '\xd0\x94\xd0\xb0': True, '\xd0\x9d\xd0\xb5\xd1\x82': False,
    'true': True, 'false': False,
    'Да': True, 'Нет': False
}

numeric_feature_value_map = {'Числовой': '"{value}"', 'Год': '"{value}"'}


def main(
    partners=('booking21', 'expedia', 'hotelscombined2', 'ostrovok'),
    lib_path='../../lib',
    partners_path='../../partners',
    resources_path='./resources',
    cached=False,
    clear_cache=False
):
    altay_db = AltayDb(use_cache=cached, clear_cache=clear_cache)
    rubrics = RUBRICS_MAP.keys()
    for rubric in rubrics:
        altay_db.load_rubric_features(rubric)

    available_features_path = join(lib_path, 'model', 'available_features.py')
    generate_available_features(available_features_path, altay_db)

    features_enums_path = join(lib_path, 'model', 'features_enums.py')
    generate_features_enums(features_enums_path, altay_db)

    hotel_features_path = join(lib_path, 'model', 'hotel_features.py')
    generate_hotel_features(hotel_features_path, altay_db)

    for partner in partners:
        logging.info("Building amenities mapping for partner {}".format(partner.title()))
        # load source table
        path = join(resources_path, '{}_amenities.csv'.format(partner))
        amenities_mapping = pd.read_csv(path, sep=';', dtype={'count': 'str', 'rubric_id': 'str', 'feature_id': 'str', 'feature_value_id': 'str'})
        amenities_mapping['amenity'] = map(lambda s: str(s).strip(), amenities_mapping['amenity'].ffill())
        amenities_mapping.sort_values(['amenity', 'feature_value'], inplace=True)
        amenities_mapping.to_csv(path, sep=';', index=None)
        amenities_mapping.reset_index(drop=True, inplace=True)

        if partner == 'expedia':
            features_mapping_path = join(partners_path, partner, 'generated', 'features_mapping.py')
        else:
            features_mapping_path = join(partners_path, partner, 'features_mapping.py')
        with GeneratedCode(features_mapping_path) as f:
            f.write('from collections import defaultdict\n\n')
            f.write('from travel.hotels.feeders.lib.model import features_enums\n\n\n')
            f.write('def create_features_mapping(hotel):\n')
            f.write('    features_mapping = defaultdict(list)\n')

            errors = []
            for i, amenity_value, feature_id, feature_value, feature_value_id in amenities_mapping[['amenity', 'feature_id', 'feature_value', 'feature_value_id']].itertuples():
                try:
                    assert amenity_value == amenity_value, "Amenity value is missing"
                    amenity_value = amenity_value.strip()
                    if feature_id != feature_id and feature_value != feature_value and feature_value_id != feature_value_id:
                        f.write('    features_mapping["{}"] += []\n'.format(AltayDb.format_dict_key(amenity_value)))
                        continue
                    assert feature_id == feature_id and feature_value == feature_value, "Feature id or value missing"
                    feature_id, feature_value = int(feature_id), feature_value.strip()
                    feature = altay_db.feature_by_id[feature_id]
                    feature_permalink = feature['permalink']
                    feature_name, value_type = AltayDb.format_feature_name(feature_permalink), feature['value_type']
                    if feature_value == '{value}':
                        value = '"{value}"'
                    elif value_type == 'logical_value':
                        assert feature_value in boolean_feature_value_map, 'Unknown logical value: {}'.format(feature_value)
                        value = boolean_feature_value_map[feature_value]
                    elif value_type == 'dict_value':
                        assert feature_value_id == feature_value_id, "Missing feature value id"
                        feature_value_id = int(feature_value_id)
                        assert feature_value_id in feature['values'], "Unknown feature value id"
                        class_name = AltayDb.format_class_name(feature_permalink)
                        feature_value_permalink = feature['values'][feature_value_id]
                        value_name = AltayDb.format_enum_name(feature_value_permalink)
                        value = 'features_enums.{}.{}'.format(class_name, value_name)
                    elif feature['value_type'] == 'number':
                        assert feature_value in numeric_feature_value_map, 'Unknown numeric value: {}'.format(feature_value)
                        value = numeric_feature_value_map[feature_value]
                    else:
                        raise Exception("Unable to process feature with value type " + value_type)
                    line = '    features_mapping["{}"] += [(hotel.{}, {})]\n'.format(AltayDb.format_dict_key(amenity_value), feature_name, value)
                    if len(line) < MAX_LINE_LEN:
                        f.write(line)
                    else:
                        parts = line.split('+=', 2)
                        f.write(parts[0] + '+=\\\n')
                        f.write(indent(2) + parts[1].strip() + '\n')
                except Exception as e:
                    errors.append((e, amenities_mapping.iloc[i]))

            logging.info("Errors count: {}".format(len(errors)))
            for e, vals in errors:
                print(e)
                print(vals)
            f.write('    return dict(features_mapping)\n')


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO, format="%(levelname)s | %(message)s")
    parser = argparse.ArgumentParser(description="Amenities Mapping Codegen")
    parser.add_argument("partners", default=['booking21', 'expedia', 'hotelscombined2', 'ostrovok'], nargs='*')
    parser.add_argument("--resources_path", default='./resources')
    parser.add_argument("--partners_path", default='../../partners')
    parser.add_argument("--lib_path", default='../../lib')
    parser.add_argument("--cached", action='store_true')
    parser.add_argument("--clear_cache", action='store_true')
    args = parser.parse_args()

    main(partners=args.partners, resources_path=args.resources_path, partners_path=args.partners_path, lib_path=args.lib_path, cached=args.cached, clear_cache=args.clear_cache)
