# -*- coding: utf8 -*-

import argparse
import logging
from collections import defaultdict
from os.path import join

import pandas as pd

from travel.hotels.feeders.lib.common.altay_db import AltayDb

UTF_HEADER = '# -*- coding: utf8 -*-\n'

CODEGEN_WARNING = """
# WARNING, THIS CODE WAS AUTOGENERATED
# DO NOT MODIFY
# TO RE-GENERATE USE $A/travel/hotels/feeders/scripts/rubric_mapping_codegen/

"""

RUBRIC_NAME_MAP = {
    30785: 'HOTEL',
    30655: 'DORMITORY',
    31632: 'HOSTEL',
    30788: 'CAMPING_AREA',
    3501492236: 'APARTMENTS_FOR_DAILY_RENT',
    31309: 'RURAL_TOURISM',
    30781: 'RESORT',
    30779: 'REST_HOUSE',
    30791: 'TOURIST_RESORT',
    30787: 'CHILDREN_CAMP',
    3501708107: 'ACCOMMODATION_FOR_DAILY_RENT',
}


def generate_mapping(path, name, values, imports):
    with open(path, 'w') as f:
        f.write(UTF_HEADER)
        f.write(CODEGEN_WARNING)
        f.write('\n'.join(imports) + '\n\n')
        f.write("{} = {{\n".format(name))
        for k, v in sorted(values.items()):
            f.write('    {}: {},\n'.format(k, v))
        f.write('}\n')


def main(
    partners=('booking21', 'dolphin', 'expedia', 'hotelscombined2', 'ostrovok', 'bronevik', 'tvil'),
    partners_path='../../partners',
    resources_path='./resources',
    cached=False,
    clear_cache=False
):
    altay_db = AltayDb(use_cache=cached, clear_cache=clear_cache)

    for partner in partners:
        logging.info("Building mapping for partner {}".format(partner.title()))
        # load source table
        path = join(resources_path, 'AccomTypes_{}.csv'.format(partner))

        table = pd.read_csv(path, sep=';', dtype={'count': 'str', 'rubric_id': 'str', 'feature_id': 'str', 'feature_value_id': 'str'})
        table['category'] = map(lambda s: str(s).strip(), table['category'])
        table.sort_values('category', inplace=True)
        table.to_csv(path, sep=';', index=None)

        if partner in ['expedia', 'tvil']:
            rubric_output_path = join(partners_path, partner, 'generated', 'rubrics_mapping.py')
        else:
            rubric_output_path = join(partners_path, partner, 'rubrics_mapping.py')
        rubric_imports = ['from travel.hotels.feeders.lib.model import enums']
        rubric_values = {}
        for category, rubric_id in zip(table['category'], table['rubric_id']):
            rubric_values['"{}"'.format(category)] = 'enums.HotelRubric.{}'.format(RUBRIC_NAME_MAP[int(rubric_id)])
        generate_mapping(path=rubric_output_path, name='rubric_map', imports=rubric_imports, values=rubric_values)

        table.dropna(subset=['feature_value_id'], inplace=True)
        if partner in ['expedia', 'tvil']:
            hotel_type_output_path = join(partners_path, partner, 'generated', 'hotel_type_mapping.py')
        else:
            hotel_type_output_path = join(partners_path, partner, 'hotel_type_mapping.py')
        hotel_type_imports = ['from travel.hotels.feeders.lib.model import features_enums']
        hotel_type_values = {}
        hotel_types_map = defaultdict(list)

        for category, feature_id, hotel_type_id in zip(table['category'], table['feature_id'], table['feature_value_id']):
            hotel_types_map[category].append((feature_id, hotel_type_id,))

        for category, type_ids in hotel_types_map.items():
            feature_enums = []
            for feature_id, feature_value_id in type_ids:
                feature_id = int(float(feature_id.strip()))
                feature_value_id = int(float(feature_value_id.strip()))

                name = 'features_enums.{}.{}'.format(
                    AltayDb.format_class_name(altay_db.features[feature_id]['permalink']),
                    AltayDb.format_enum_name(get_feature_map(altay_db, feature_id)[feature_value_id])
                )
                feature_enums.append(name)

            hotel_type_values['"{}"'.format(category)] = '[{}]'.format(', '.join(sorted(feature_enums)))

        generate_mapping(path=hotel_type_output_path, name='hotel_type_map', imports=hotel_type_imports, values=hotel_type_values)


def get_feature_map(altay_db, feature_id):
    return next(altay_db.extract_feature(altay_db.features[feature_id]))['values']


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(levelname)s | %(message)s")
    parser = argparse.ArgumentParser(description="Rubric Mapping Codegen")
    parser.add_argument("partners", default=['booking21', 'dolphin', 'expedia', 'hotelscombined2', 'ostrovok'], nargs='*')
    parser.add_argument("--resources_path", default='./resources')
    parser.add_argument("--partners_path", default='../../partners')
    parser.add_argument("--cached", action='store_true')
    parser.add_argument("--clear_cache", action='store_true')
    args = parser.parse_args()

    main(partners=args.partners, partners_path=args.partners_path, resources_path=args.resources_path, cached=args.cached, clear_cache=args.clear_cache)
