#!/usr/bin/env python

import argparse
import contextlib
import itertools
import json
import logging
import os
import random
import re
import subprocess
import sys
import time

import requests


def teamcity_escape(s):
    s = re.sub("(['\\[\\]|])", "|\\1", s)
    s = s.replace("\n", "|n").replace("\r", "|r")
    s = "'" + s + "'"
    return s


def teamcity_interact(message, *args, **kwargs):
    r = " ".join(itertools.chain(
        [message],
        (
            teamcity_escape(str(x))
            for x in args
        ),
        (
            "{0}={1}".format(str(k), teamcity_escape(str(v)))
            for k, v in kwargs.iteritems())
    ))
    r = "##teamcity[" + r + "]\n"
    sys.stdout.flush()
    sys.stderr.write(r)
    sys.stderr.flush()


def teamcity_message(text, status="NORMAL"):
    if status not in ["NORMAL", "WARNING", "FAILURE", "ERROR"]:
        raise ValueError("Invalid |status|: {0}".format(status))

    if status == "NORMAL":
        teamcity_interact("message", text=text)
    else:
        teamcity_interact("message", text=text, status=status)


def teamcity_set_parameter(name, value):
    teamcity_interact("setParameter", name=name, value=value)


@contextlib.contextmanager
def teamcity_block(name):
    try:
        teamcity_interact("blockOpened", name=name)
        yield
    finally:
        teamcity_interact("blockClosed", name=name)


def die(msg):
    logging.fatal(msg)
    teamcity_interact("buildProblem", description=msg)
    sys.exit(1)


def main():
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(levelname)s | %(message)s", stream=sys.stdout)

    parser = argparse.ArgumentParser()
    released_choices = ['testing', 'stable']
    parser.add_argument("--teamcity-btid", type=str, required=True)
    parser.add_argument("--teamcity-bid", type=str, required=True)
    parser.add_argument("--teamcity-build-number", type=str, required=True)
    parser.add_argument("--sandbox-retries", type=int, default=5)
    parser.add_argument("--feeder", type=str, required=True)
    parser.add_argument("--sandbox-package-type", type=str, default="TRAVEL_HOTELS_FEEDERS",
                        help="Package type, generally can be found in completed sandbox build task attributes "
                             "and arcadia/sandbox/projects/Travel/resources/__init__.py "
                             "e.g. YA_PACKAGE, TRAVEL_HOTELS_FEEDERS etc.")
    parser.add_argument('--released-at', choices=released_choices, default='stable')
    parser.add_argument("passthrough", nargs=argparse.REMAINDER)
    args = parser.parse_args()

    logging.info("teamcity build id: %s", args.teamcity_bid)
    logging.info("teamcity build type id: %s", args.teamcity_btid)
    logging.info("teamcity build number: %s", args.teamcity_build_number)

    teamcity_url = "https://teamcity.yandex-team.ru/viewLog.html?buildId={}&buildTypeId={}".format(
        args.teamcity_bid, args.teamcity_btid)

    sandbox = os.path.join(os.getcwd(), "__".join([args.teamcity_btid, args.teamcity_build_number]))
    if os.path.exists(sandbox):
        die("sandbox already exists")

    logging.info("sandbox directory: %s", sandbox)
    os.mkdir(sandbox)
    os.chdir(sandbox)

    with teamcity_block("querying resource"):
        params = {"limit": 1, "offset": 0,
                  "type": args.sandbox_package_type, "state": "READY",
                  "attrs": json.dumps({"resource_name": "travel.hotels.feeders", "released": args.released_at})}
        headers = {"accept": "application/json; charset=utf-8"}

        tries = 0
        delay = 3.0
        while True:
            rsp = requests.get("https://sandbox.yandex-team.ru/api/v1.0/resource", params=params, headers=headers)
            if rsp.ok:
                break

            logging.error("failed to query sandbox api (status_code=%s; body=%s)", rsp.status_code, rsp.text)
            tries += 1
            delay *= random.uniform(1.5, 2.5)
            if tries < args.sandbox_retries:
                logging.info("sleeping for %.2fs", delay)
                time.sleep(delay)
            else:
                die("sandbox api is unavailable")

        rsp = rsp.json()
        if len(rsp.get("items", [])) == 0:
            die("no matching resources found")

        resource = rsp["items"][0]

        resource_id = resource["id"]
        resource_url = "https://sandbox.yandex-team.ru/resource/%s/" % resource_id
        task_id = resource["task"]["id"]
        task_url = "https://sandbox.yandex-team.ru/task/%s/" % task_id

        logging.info("resource #%s (%s)", resource_id, resource_url)
        logging.info("task #%s (%s)", resource["task"]["id"], task_url)

        teamcity_set_parameter("sandbox.resource.id", resource_id)
        teamcity_set_parameter("sandbox.resource.url", resource_url)
        teamcity_set_parameter("sandbox.task.id", task_id)
        teamcity_set_parameter("sandbox.task.url", task_url)

    with teamcity_block("downloading resource"):
        subprocess.check_call(["wget", "--progress=dot:giga", "-O", "resource.tar.gz", resource["http"]["proxy"]])

    with teamcity_block("unpacking resource"):
        subprocess.check_call(["tar", "zxvf", "resource.tar.gz"])

        with open("build.branch.txt") as handle:
            build_branch = handle.read().strip()
            teamcity_set_parameter("resource.build.branch", build_branch)

        with open("build.revision.txt") as handle:
            build_revision = handle.read().strip()
            teamcity_set_parameter("resource.build.revision", build_revision)

        with open("build.sandbox_task_id.txt") as handle:
            build_sandbox_task_id = handle.read().strip()
            teamcity_set_parameter("resource.build.sandbox_task_id", build_sandbox_task_id)

    child_args = [
        "./%s/%s" % (args.feeder, args.feeder),
        "--enable-teamcity-diagnostics",
        "--annotate", "build_branch={}".format(build_branch),
        "--annotate", "build_revision={}".format(build_revision),
        "--annotate", "build_sandbox_task_id={}".format(build_sandbox_task_id),
        "--annotate", "teamcity_build_id={}".format(args.teamcity_bid),
        "--annotate", "teamcity_build_type_id={}".format(args.teamcity_btid),
        "--annotate", "teamcity_build_number={}".format(args.teamcity_build_number),
        "--annotate", "teamcity_url={}".format(teamcity_url),
    ]
    child_args.extend(filter(lambda arg: arg != "--" and arg != "", args.passthrough))

    os.execv(child_args[0], child_args)


if __name__ == "__main__":
    main()
