from __future__ import print_function

import argparse
import difflib
import json
import os
from os.path import join

import library.python.resource as rs

from travel.hotels.feeders.scripts.update_altay_configs.editor_client import EditorClient
from travel.hotels.feeders.scripts.update_altay_configs.generate_feed_config import generate_feed_config


def get_config_diff(new_config, feed_id, client):
    old_config = client.get_feed(feed_id=feed_id)
    del old_config['provider']
    # del old_config['id']
    del old_config['commit_id']

    new_config = new_config.copy()
    del new_config['provider_id']
    # del config['id']
    config_diff = difflib.ndiff(json.dumps(old_config, indent=2).split('\n'), json.dumps(new_config, indent=2).split('\n'))
    return [l for l in config_diff if l.startswith('+ ') or l.startswith('- ')]


def commit_config(config, settings, client):
    config_text = json.dumps(config)
    if 'id' not in settings:
        ans = raw_input("No feed \"id\" in config.\nWill create new feed.\nThis is irreversible. Continue? (y/N)")
        if ans != 'y':
            print("Skipping: cancelled by user")
        return client.add_feed(config_text)
    else:
        changes = get_config_diff(config, settings['id'], client)
        if changes:
            print("Changes:")
            print('\n'.join(changes))
            ans = raw_input("Continue? (y/N)")
            if ans != 'y':
                print("Skipping: cancelled by user")
                return
            return client.modify_feed(settings['id'], config_text)
        else:
            print("Skipping: no changes")


def get_settings_files(settings_paths):
    if not isinstance(settings_paths, list):
        settings_paths = [settings_paths]
    for path in settings_paths:
        if os.path.isdir(path):
            for name in os.listdir(path):
                if name.endswith('.json'):
                    yield join(path, name)
        else:
            yield path


def get_settings(settings_paths):
    for path in get_settings_files(settings_paths):
        with open(path, 'r') as f:
            data = json.load(f)
            if isinstance(data, list):
                for item in data:
                    yield item
            else:
                yield data
        with open(path, 'w') as f:
            json.dump(data, f, indent=2, sort_keys=True)


def show(settings_paths, yt_token=None, verbose=False, **kwargs):
    default_settings = json.loads(rs.find('/default_settings.json'))
    for feed_settings in get_settings(settings_paths):
        settings = default_settings.copy()
        settings.update(feed_settings)
        config = generate_feed_config(yt_token=yt_token, **settings)
        verbose and print("Feed {}:".format(config['description']))
        print(json.dumps(config, indent=2))


def save(settings_paths, output_dir, yt_token=None, verbose=False, **kwargs):
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)
    default_settings = json.loads(rs.find('/default_settings.json'))
    for feed_settings in get_settings(settings_paths):
        settings = default_settings.copy()
        settings.update(feed_settings)
        config = generate_feed_config(yt_token=yt_token, **settings)
        output_path = join(output_dir, '{}_{}.json'.format(settings['feed_name'], settings['feed_id']))
        with open(output_path, 'w') as f:
            json.dump(config, f, indent=2)
        verbose and print("Saved {}".format(output_path))


def diff(settings_paths, uid, yt_token=None, debug=False, **kwargs):
    altay_client = EditorClient(uid=uid, prod=not debug)
    default_settings = json.loads(rs.find('/default_settings.json'))
    for feed_settings in get_settings(settings_paths):
        settings = default_settings.copy()
        settings.update(feed_settings)
        config = generate_feed_config(yt_token=yt_token, **settings)
        print("Feed {}".format(config['description']))
        if 'id' in settings:
            changes = get_config_diff(new_config=config, feed_id=settings['id'], client=altay_client)
            print('\n'.join(changes))
        else:
            print("Nothing to compare with: no feed \"id\"")


def commit(settings_paths, uid, yt_token=None, debug=False, **kwargs):
    altay_client = EditorClient(uid=uid, prod=not debug)
    default_settings = json.loads(rs.find('/default_settings.json'))
    for feed_settings in get_settings(settings_paths):
        settings = default_settings.copy()
        settings.update(feed_settings)
        config = generate_feed_config(yt_token=yt_token, **settings)
        print("Submitting config {}".format(config['description']))
        result = commit_config(config=config, settings=settings, client=altay_client)
        if result is None:
            continue
        print("Config submitted, result: {}".format(result))
        if 'id' not in settings:
            # will be saved in original file
            feed_settings['id'] = result['id']


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('settings_paths', default=['./partners_settings'], nargs='?', help="Settings for config generation. For feed settings details see README")
    parser.add_argument('--debug', action='store_true', help='use testing altay client')
    parser.add_argument('--verbose', '-v', action='store_true', help='Verbose mode')
    parser.add_argument('--yt_token', default=None, help='Token is used for reading fields list from feeds tables on YT')
    parser.add_argument('--yt_proxy', default='hahn', help='Token is used for reading fields list from feeds tables on YT')

    subparsers = parser.add_subparsers()
    show_parser = subparsers.add_parser('show', help='print generated configs')
    show_parser.set_defaults(func=show)

    save_parser = subparsers.add_parser('save', help='save generated configs to disk')
    save_parser.add_argument('--output_dir', default='./output')
    save_parser.set_defaults(func=save)

    diff_parser = subparsers.add_parser('diff', help='compare generated configs with the ones already in altay')
    diff_parser.add_argument('--uid', '--altay_uid', type=int, required=True, help="Your uid for altay client. From https://staff-api.yandex-team.ru/v3/persons?login={user}")
    diff_parser.set_defaults(func=diff)

    commit_parser = subparsers.add_parser('commit', help='commit configs to altay')
    commit_parser.add_argument('--uid', '--altay_uid', type=int, required=True, help="Your uid for altay client. From https://staff-api.yandex-team.ru/v3/persons?login={user}")
    commit_parser.set_defaults(func=commit)

    args = parser.parse_args()
    args.func(**args.__dict__)
