import json
from collections import OrderedDict

import requests
import six


class EditorClient(object):
    prod_url = "http://editor.altay-api.search.yandex.net"
    testing_url = "http://rc-editor.altay-api.search.yandex.net"

    def __init__(self, uid, prod, api_version='v0.2'):
        self.uid = uid
        self.prod = prod
        self.api_version = api_version

    def _send_request(self, target, method, http_method, body=None, **kwargs):
        base_url = self.prod_url if self.prod else self.testing_url
        kwargs['user'] = self.uid
        url = "{base_url}/{api_version}/{target}/{method}".format(
            base_url=base_url,
            api_version=self.api_version,
            target=target,
            method=method
        )
        body = json.loads(body) if isinstance(body, six.string_types) else body
        response = requests.request(method=http_method, url=url, params=kwargs, json=body)
        return response

    def get_feed(self, feed_id, commit_id=None):
        response = self._send_request('feed', 'get', 'GET', id=feed_id, **{'commit-id': commit_id})
        if response.status_code == 200:
            return json.loads(response.text, object_pairs_hook=OrderedDict)
        else:
            raise Exception(response.status_code, response.text)

    def add_feed(self, feed_config):
        response = self._send_request('feed', 'add', 'POST', body=feed_config)
        if response.status_code == 200:
            return response.json()
        else:
            raise Exception(response.status_code, response.text)

    def modify_feed(self, feed_id, feed_config):
        response = self._send_request('feed', 'modify', 'POST', id=feed_id, body=feed_config)
        if response.status_code == 200:
            return response.json()
        else:
            raise Exception(response.status_code, response.text)
