# -*- coding: utf8 -*-
import json
from collections import OrderedDict

from travel.hotels.lib.python import ytlib

import library.python.resource as rs

URI_TEMPLATE = "https://hahn.yt.yandex-team.ru/api/v3/read_table?path={table_path}&output_format[$value]=yson&output_format[$attributes][format]=binary"


def generate_feed_config(
    enabled,  # whether or not feed is to be uploaded to Altay and used in prod
    feed_name,  # required, feed dir name hotelbook_direct
    feed_id,  # required
    provider_id,
    is_russia,
    is_hotels,
    id=None,
    publish_photos=None,
    illegal=False,
    custom_file_columns_mapping=None,
    is_draft=False,
    discard_rubric=False,
    discard_phone=None,
    trust_coordinates=None,
    trust_address=None,  # true if world, false if russia
    geocode_by_coordinates=None,
    emails=("svyatokum@yandex-team.ru", "travel-alerts@yandex-team.ru"),
    yt_token=None,
    yt_proxy="hahn",
    **kwargs
):

    ytlib.configure(yt_proxy, yt_token)
    feed_table = '_'.join(['russia' if is_russia else 'world', 'hotels' if is_hotels else 'apartments'])
    table_path = '//home/travel/prod/feeds/{feed_name}/latest/parsed/{feed_table}'.format(feed_name=feed_name, feed_table=feed_table)

    # get list of columns from feed table
    fields_list = [field['name'] for field in ytlib.yt.get_attribute(table_path, 'schema')]

    config = json.loads(rs.find('/template.json'), object_pairs_hook=OrderedDict)

    config['description'] = ' '.join([feed_name.title().replace('_', ' '), 'Russia' if is_russia else 'World', 'Hotels' if is_hotels else 'Apartments'])
    config['feed_id'] = feed_id
    config['provider_id'] = provider_id
    config['id'] = id
    config['status'] = "enabled" if enabled else "deleted"
    config['uri'] = URI_TEMPLATE.format(table_path=table_path)

    config['legality'] = "illegal" if illegal else "legal"
    # feed fields:
    file_columns_mapping = {field: "unused" if field.startswith('_') else field for field in fields_list}
    file_columns_mapping['country'] = "unused"
    file_columns_mapping.update(custom_file_columns_mapping or {})
    config['lang'] = 'ru' if is_russia else 'en'

    config['merging_settings']['attributes'] = []
    if trust_address is None:
        trust_address = not is_russia
    if trust_address:
        config['merging_settings']['attributes'].append({
            "attribute": "address_attribute",
            "exclusivity": False,
            "trust_level": "trusted"
        })
    config["unification_settings"]['use_source_address_text'] = trust_address
    if trust_coordinates is None:
        trust_coordinates = True  # not is_russia
    if trust_coordinates:
        config['merging_settings']['attributes'].append({
            "attribute": "coordinates_attribute",
            "exclusivity": False,
            "trust_level": "trusted"
        })
    config["unification_settings"]['use_manual_coordinates'] = trust_coordinates
    if discard_rubric:
        config['merging_settings']['attributes'].append({
            "attribute": "rubric_attribute",
            "exclusivity": False,
            "trust_level": "incorrect"
        })
    if discard_phone is None:
        discard_phone = not is_hotels
    if 'phone' in file_columns_mapping and discard_phone:
        config['merging_settings']['attributes'].append({
            "attribute": "phone_attribute",
            "exclusivity": False,
            "trust_level": "incorrect"
        })
    if publish_photos is None:
        publish_photos = is_russia
    if 'photos' in file_columns_mapping and not publish_photos:
        file_columns_mapping['photos'] = "unused"
    if geocode_by_coordinates is None:
        geocode_by_coordinates = True  # todo: consider trusting geoconot is_russia
    config['feed_parsing_settings']['file_columns_mapping'] = [OrderedDict((('in_file', k), ('in_code', v))) for k, v in sorted(file_columns_mapping.items(), key=lambda x: x[0], reverse=True)]
    config["unification_settings"]['address_unification_mode'] = 'only_by_coordinates' if geocode_by_coordinates else 'default'
    if is_draft:
        config['status'] = "draft"
    config['emails'] = emails

    return config
